import { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { supabase } from '../supabaseClient';
import { Button } from '@/components/ui/button';
import { FaUsers, FaEnvelope, FaCog } from 'react-icons/fa';
import { LogOut as Signout } from 'lucide-react';
import { Input } from '@/components/ui/input';

// Define the Organization type
interface Organization {
  id: string;
  name: string;
  members?: any[]; // Adjust type if you have a specific member type
}

export default function OrganizationDashboard() {
  const [organization, setOrganization] = useState<Organization | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [activeTab, setActiveTab] = useState<'members' | 'invites' | 'settings' | 'dashboard'>('members');
  const navigate = useNavigate();

  const handleSignOut = async () => {
    try {
      const { error } = await supabase.auth.signOut();
      if (error) throw error;
      window.location.href = "/Auth";
    } catch (err) {
      console.error("Error signing out:", err);
    }
  };

  const fetchOrganization = async () => {
    try {
      setLoading(true);
      setError(null);

      // 1. Get current user
      const { data: { user }, error: userError } = await supabase.auth.getUser();
      if (userError || !user) {
        throw new Error(userError?.message || 'Not authenticated');
      }

      // 2. Check if user owns any organization
      const { data: ownedOrg } = await supabase
        .from('organizations')
        .select('*')
        .eq('owner_id', user.id)
        .maybeSingle();

      if (ownedOrg) {
        setOrganization(ownedOrg);
        return;
      }

      // 3. Check if user is a member
      const { data: membership } = await supabase
        .from('organization_members')
        .select('organization_id')
        .eq('user_id', user.id)
        .limit(1);

      if (!membership || membership.length === 0) {
        throw new Error('No organization found');
      }

      // 4. Get the organization data
      const { data: memberOrg, error: orgError } = await supabase
        .from('organizations')
        .select('*')
        .eq('id', membership[0].organization_id)
        .single();

      if (orgError || !memberOrg) {
        throw new Error(orgError?.message || 'Organization not found');
      }

      setOrganization(memberOrg);

    } catch (err: any) {
      console.error('Error fetching organization:', err);
      setError(err instanceof Error ? err.message : 'An unknown error occurred');

      if (err.message?.includes('No organization found')) {
        navigate('/CreateOrganization');
      }
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchOrganization();
  }, []);

  if (loading) {
    return (
      <div className="flex justify-center items-center min-h-screen">
        <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-indigo-500"></div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="error-container p-8 max-w-md mx-auto text-center">
        <h2 className="text-2xl font-bold mb-4">Error</h2>
        <p className="mb-6">{error}</p>
        <div className="space-y-3">
          <Button onClick={() => window.location.reload()} className="w-full">Retry</Button>
          <Button variant="outline" onClick={() => navigate('/')} className="w-full">
            Return Home
          </Button>
        </div>
      </div>
    );
  }

  if (!organization) {
    return (
      <div className="no-organization p-8 max-w-md mx-auto text-center">
        <h2 className="text-2xl font-bold mb-4">No Organization Found</h2>
        <Button onClick={() => navigate('/CreateOrganization')} className="w-full">
          Create New Organization
        </Button>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <header className="bg-white shadow">
        <div className="max-w-7xl mx-auto px-4 py-6 sm:px-6 lg:px-8 flex justify-between items-center">
          <h1 className="text-3xl font-bold text-gray-900">{organization.name}</h1>
          <Button variant="outline" onClick={handleSignOut}>
            <Signout className="mr-2 h-4 w-4" /> Sign Out
          </Button>
        </div>
      </header>

      {/* Main Content */}
      <main className="max-w-7xl mx-auto px-4 py-6 sm:px-6 lg:px-8">
        {/* Tabs */}
        <div className="border-b border-gray-200">
          <nav className="-mb-px flex space-x-8">
            <button
              onClick={() => setActiveTab('members')}
              className={`${activeTab === 'members'
                ? 'border-indigo-500 text-indigo-600'
                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                } whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm`}
            >
              <FaUsers className="inline mr-2" /> Members
            </button>
            <button
              onClick={() => setActiveTab('invites')}
              className={`${activeTab === 'invites'
                ? 'border-indigo-500 text-indigo-600'
                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                } whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm`}
            >
              <FaEnvelope className="inline mr-2" /> Invites
            </button>
            <button
              onClick={() => setActiveTab('settings')}
              className={`${activeTab === 'settings'
                ? 'border-indigo-500 text-indigo-600'
                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                } whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm`}
            >
              <FaCog className="inline mr-2" /> Settings
            </button>
            <button
              onClick={() => navigate('/DTTDashboard')}
              className={`${activeTab === 'dashboard'
                ? 'border-indigo-500 text-indigo-600'
                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                } whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm`}
            >
              <FaCog className="inline mr-2" /> Dashboard
            </button>
          </nav>
        </div>

        {/* Tab Content */}
        <div className="mt-6">
          {activeTab === 'members' && <MembersTab organization={organization} />}
          {activeTab === 'invites' && <InvitesTab organization={organization} />}
          {activeTab === 'settings' && <SettingsTab organization={organization} />}
          {activeTab === 'dashboard' && navigate('/DTTDashboard')}
        </div>
      </main>
    </div>
  );
}

// --- Members Tab ---
function MembersTab({ organization }: { organization: Organization }) {
  return (
    <div>
      <h2 className="text-xl font-semibold mb-4">Organization Members</h2>
      <div className="bg-white shadow overflow-hidden sm:rounded-lg">
        <table className="min-w-full divide-y divide-gray-200">
          <thead className="bg-gray-50">
            <tr>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Name</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Email</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Role</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
              <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
            </tr>
          </thead>
          <tbody className="bg-white divide-y divide-gray-200">
            {organization.members?.map((member: any) => (
              <tr key={member.id}>
                <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">{member.name}</td>
                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">{member.email}</td>
                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">{member.role}</td>
                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                  <span className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${
                    member.status === 'active' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'
                  }`}>
                    {member.status}
                  </span>
                </td>
                <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                  <button className="text-indigo-600 hover:text-indigo-900 mr-3">Edit</button>
                  <button className="text-red-600 hover:text-red-900">Remove</button>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
    </div>
  );
}

// --- Invites Tab ---
function InvitesTab({ organization }: { organization: Organization }) {
  const [email, setEmail] = useState('')
  const [role, setRole] = useState<'admin' | 'hr' | 'member'>('member')
  const [loading, setLoading] = useState(false)
  const [message, setMessage] = useState('')
  const [status, setStatus] = useState<'pending' | 'approved' | 'error'>('pending')

  const sendInvite = async () => {
    setLoading(true);
    setMessage('');
    
    if (!email) {
      setMessage('Please enter a valid email address.');
      setLoading(false);
      return;
    }

    try {
      // Check if already invited
      const { data: existingInvite, error: checkError } = await supabase
        .from("organization_invites")
        .select("*")
        .eq("organization_id", organization.id)
        .eq("email", email)
        .maybeSingle();

      if (checkError) throw checkError;
      if (existingInvite) {
        setMessage("This email has already been invited to this organization.");
        return;
      }

      // Send new invite
      const { error } = await supabase
        .from('organization_invites')
        .insert([{ 
          organization_id: organization.id, 
          email, 
          role, 
          status: 'pending' 
        }]);

      if (error) throw error;

      setMessage(`Invite sent to ${email}`);
      setEmail('');
      setRole('member');
    } catch (error: any) {
      console.error('Error sending invite:', error);
      setMessage(`Failed to send invite: ${error.message}`);
    } finally {
      setLoading(false);
    }
  };

  return (
    <div>
      <h2 className="text-xl font-semibold mb-4">Invite New Members</h2>
      <div className="bg-white shadow p-6 rounded-lg">
        <div className="mb-4">
          <label className="block text-sm font-medium text-gray-700 mb-1">Email Address</label>
          <Input
            type="email"
            value={email}
            onChange={(e) => setEmail(e.target.value)}
            placeholder="member@example.com"
            required
            className="w-full"
          />
        </div>

        <div className="mb-4">
          <label className="block text-sm font-medium text-gray-700 mb-1">Role</label>
          <select
            value={role}
            onChange={(e) => setRole(e.target.value as any)}
            className="w-full p-2 border rounded-md border-gray-300 focus:border-indigo-500 focus:ring-indigo-500"
          >
            <option value="admin">Admin</option>
            <option value="hr">HR</option>
            <option value="member">Member</option>
          </select>
        </div>

        <Button onClick={sendInvite} disabled={loading}>
          {loading ? 'Sending...' : 'Send Invite'}
        </Button>

        {message && (
          <div className={`mt-4 p-3 rounded-md ${
            message.includes('sent') 
              ? 'bg-green-50 text-green-700' 
              : 'bg-red-50 text-red-700'
          }`}>
            {message}
          </div>
        )}
      </div>
    </div>
  );
}

// --- Settings Tab ---
function SettingsTab({ organization }: { organization: Organization }) {
  const [name, setName] = useState(organization.name);
  const [loading, setLoading] = useState(false);
  const [message, setMessage] = useState('');

  const handleSave = async () => {
    setLoading(true);
    try {
      const { error } = await supabase
        .from('organizations')
        .update({ name })
        .eq('id', organization.id);
      
      if (error) throw error;
      setMessage('Organization settings updated successfully');
    } catch (error: any) {
      setMessage(error.message);
    } finally {
      setLoading(false);
    }
  };

  return (
    <div>
      <h2 className="text-xl font-semibold mb-4">Organization Settings</h2>
      <div className="bg-white shadow p-6 rounded-lg">
        <div className="mb-4">
          <label className="block text-sm font-medium text-gray-700 mb-1">Organization Name</label>
          <Input
            type="text"
            value={name}
            onChange={(e) => setName(e.target.value)}
            className="w-full"
          />
        </div>
        <Button onClick={handleSave} disabled={loading}>
          {loading ? 'Saving...' : 'Save Changes'}
        </Button>
        {message && (
          <p className={`mt-2 text-sm ${
            message.includes('successfully') 
              ? 'text-green-600' 
              : 'text-red-600'
          }`}>
            {message}
          </p>
        )}
      </div>
    </div>
  );
} 