// AcceptInvite.tsx
import { useEffect, useState } from 'react'
import { useParams, useNavigate } from 'react-router-dom'
import { supabase } from '../supabaseClient'
import { Button } from '@/components/ui/button'

export default function AcceptInvite() {
  const { token } = useParams()
  const navigate = useNavigate()
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState('')
  const [invite, setInvite] = useState<any>(null)

  useEffect(() => {
    const verifyInvite = async () => {
      try {
        const { data, error } = await supabase
          .from('organization_invites')
          .select('*')
          .eq('token', token)
          .single()

        if (error) throw error
        if (!data) throw new Error('Invalid invitation token')
        if (data.status !== 'pending') throw new Error('Invitation already used')
        if (new Date(data.expires_at) < new Date()) {
          throw new Error('Invitation has expired')
        }

        setInvite(data)
      } catch (err: any) {
        setError(err.message)
      } finally {
        setLoading(false)
      }
    }

    verifyInvite()
  }, [token])

  const acceptInvite = async () => {
    try {
      setLoading(true)
      const { data: { user }, error: userError } = await supabase.auth.getUser()
      
      if (userError || !user) {
        // Redirect to sign up if not logged in
        navigate(`/signup?invite_token=${token}`)
        return
      }

      // Add user to organization
      const { error: memberError } = await supabase
        .from('organization_members')
        .insert([{
          organization_id: invite.organization_id,
          user_id: user.id,
          user_email: user.email,
          role: invite.role,
          status: 'active'
        }])

      if (memberError) throw memberError

      // Update invite status
      await supabase
        .from('organization_invites')
        .update({ status: 'accepted' })
        .eq('id', invite.id)

      navigate(`/organization/${invite.organization_id}`)
    } catch (err: any) {
      setError(err.message)
    } finally {
      setLoading(false)
    }
  }

  if (loading) return <div>Loading...</div>
  if (error) return <div className="error-message">{error}</div>

  return (
    <div className="max-w-md mx-auto p-6">
      <h2 className="text-2xl font-bold mb-4">Organization Invitation</h2>
      <p className="mb-6">You've been invited to join {invite.organizations.name} as a {invite.role}.</p>
      
      <Button 
        onClick={acceptInvite} 
        disabled={loading}
        className="w-full"
      >
        {loading ? 'Accepting...' : 'Accept Invitation'}
      </Button>
    </div>
  )
}