import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { supabase } from '@/components/supabaseClient';
import { Loader2, User, ClipboardList, AlertCircle, Calendar, TrendingUp } from 'lucide-react';

interface AssignTaskFormProps {
  onSuccess: () => void;
  onCancel: () => void;
}

interface Task {
  id: string;
  name: string;
  description: string;
  due_date: string;
  status: string;
  priority: string;
  assigned_to: string | null;
  assigned_to_name?: string;
  project_name?: string;
}

interface TeamMember {
  id: string;
  name: string;
  email: string;
  role: string;
  team_id: string;
  current_tasks_count?: number;
}

interface Team {
  id: string;
  name: string;
}

export function AssignTaskForm({ onSuccess, onCancel }: AssignTaskFormProps) {
  const [formData, setFormData] = useState({
    task_id: '',
    assigned_to: ''
  });
  const [tasks, setTasks] = useState<Task[]>([]);
  const [teamMembers, setTeamMembers] = useState<TeamMember[]>([]);
  const [teams, setTeams] = useState<Team[]>([]);
  const [isLoading, setIsLoading] = useState(false);
  const [isFetching, setIsFetching] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [selectedTeam, setSelectedTeam] = useState<string>('');
  const [selectedTask, setSelectedTask] = useState<Task | null>(null);
  const [selectedMember, setSelectedMember] = useState<TeamMember | null>(null);

  useEffect(() => {
    fetchData();
  }, []);

  useEffect(() => {
    if (selectedTeam) {
      fetchTeamMembers(selectedTeam);
    }
  }, [selectedTeam]);

  useEffect(() => {
    if (formData.task_id) {
      const task = tasks.find(t => t.id === formData.task_id);
      setSelectedTask(task || null);
    }
  }, [formData.task_id, tasks]);

  useEffect(() => {
    if (formData.assigned_to) {
      const member = teamMembers.find(m => m.id === formData.assigned_to);
      setSelectedMember(member || null);
    } else {
      setSelectedMember(null);
    }
  }, [formData.assigned_to, teamMembers]);

  const fetchData = async () => {
    try {
      setIsFetching(true);
      setError(null);

      // Fetch unassigned tasks or tasks that can be reassigned
      const { data: tasksData, error: tasksError } = await supabase
        .from('tasks')
        .select(`
          id,
          name,
          description,
          due_date,
          status,
          priority,
          assigned_to,
          projects (name)
        `)
        .order('created_at', { ascending: false });

      // Fetch all teams
      const { data: teamsData, error: teamsError } = await supabase
        .from('teams')
        .select('id, name')
        .order('name');

      if (tasksError) throw tasksError;
      if (teamsError) throw teamsError;

      // Transform tasks data
      const transformedTasks = (tasksData || []).map(task => ({
        ...task,
        project_name: task.projects?.name || 'No project'
      }));

      setTasks(transformedTasks);
      setTeams(teamsData || []);

      // Select first team by default
      if (teamsData && teamsData.length > 0) {
        setSelectedTeam(teamsData[0].id);
        await fetchTeamMembers(teamsData[0].id);
      }

    } catch (err) {
      console.error('Error fetching data:', err);
      setError('Failed to load data. Please try again.');
    } finally {
      setIsFetching(false);
    }
  };

  const fetchTeamMembers = async (teamId: string) => {
    try {
      const { data: membersData, error: membersError } = await supabase
        .from('team_members')
        .select('id, name, email, role, team_id')
        .eq('team_id', teamId)
        .order('name');

      if (membersError) throw membersError;

      // Fetch task counts for each member
      const membersWithTaskCounts = await Promise.all(
        (membersData || []).map(async (member) => {
          const { count, error: countError } = await supabase
            .from('tasks')
            .select('*', { count: 'exact' })
            .eq('assigned_to', member.id)
            .in('status', ['pending', 'in_progress']);

          if (countError) {
            console.error('Error fetching task count:', countError);
            return { ...member, current_tasks_count: 0 };
          }

          return { ...member, current_tasks_count: count || 0 };
        })
      );

      setTeamMembers(membersWithTaskCounts);
    } catch (err) {
      console.error('Error fetching team members:', err);
      setError('Failed to load team members.');
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.task_id || !formData.assigned_to) {
      setError('Please select both a task and a team member');
      return;
    }

    setIsLoading(true);
    setError(null);

    try {
      const { error: updateError } = await supabase
        .from('tasks')
        .update({ 
          assigned_to: formData.assigned_to,
          updated_at: new Date().toISOString()
        })
        .eq('id', formData.task_id);

      if (updateError) throw updateError;

      console.log('Task assigned successfully');
      onSuccess();
      
    } catch (err) {
      console.error('Error assigning task:', err);
      setError('Failed to assign task. Please try again.');
    } finally {
      setIsLoading(false);
    }
  };

  const handleTaskChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    setFormData({ ...formData, task_id: e.target.value });
    setError(null);
  };

  const handleMemberChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    setFormData({ ...formData, assigned_to: e.target.value });
    setError(null);
  };

  const handleTeamChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    setSelectedTeam(e.target.value);
    setFormData({ ...formData, assigned_to: '' }); // Reset assigned member when team changes
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString();
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'high': return 'text-red-500';
      case 'medium': return 'text-yellow-500';
      case 'low': return 'text-green-500';
      default: return 'text-gray-500';
    }
  };

  if (isFetching) {
    return (
      <div className="flex flex-col items-center justify-center py-8">
        <Loader2 className="w-8 h-8 animate-spin text-primary mb-4" />
        <p className="text-muted-foreground">Loading tasks and teams...</p>
      </div>
    );
  }

  return (
    <form onSubmit={handleSubmit} className="space-y-6">
      <div>
        <h3 className="text-lg font-medium text-foreground mb-4">
          Assign Task to Team Member
        </h3>
        
        {error && (
          <div className="bg-destructive/15 text-destructive p-3 rounded-md mb-4 flex items-center">
            <AlertCircle className="w-4 h-4 mr-2" />
            <p className="text-sm">{error}</p>
          </div>
        )}

        <div className="space-y-4">
          {/* Task Selection */}
          <div>
            <label className="block text-sm font-medium text-foreground mb-2">
              Select Task
            </label>
            <div className="relative">
              <ClipboardList className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-muted-foreground" />
              <select
                required
                value={formData.task_id}
                onChange={handleTaskChange}
                className="w-full pl-10 p-3 border border-border rounded-md bg-background text-foreground focus:ring-2 focus:ring-primary focus:border-transparent"
              >
                <option value="">Choose a task...</option>
                {tasks.map((task) => (
                  <option key={task.id} value={task.id}>
                    {task.name} {task.assigned_to ? '(Already assigned)' : ''}
                  </option>
                ))}
              </select>
            </div>
          </div>

          {/* Team Selection */}
          <div>
            <label className="block text-sm font-medium text-foreground mb-2">
              Select Team
            </label>
            <div className="relative">
              <User className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-muted-foreground" />
              <select
                value={selectedTeam}
                onChange={handleTeamChange}
                className="w-full pl-10 p-3 border border-border rounded-md bg-background text-foreground focus:ring-2 focus:ring-primary focus:border-transparent"
              >
                {teams.map((team) => (
                  <option key={team.id} value={team.id}>
                    {team.name}
                  </option>
                ))}
              </select>
            </div>
          </div>

          {/* Team Member Selection */}
          <div>
            <label className="block text-sm font-medium text-foreground mb-2">
              Assign to Team Member
            </label>
            <div className="relative">
              <User className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-muted-foreground" />
              <select
                required
                value={formData.assigned_to}
                onChange={handleMemberChange}
                className="w-full pl-10 p-3 border border-border rounded-md bg-background text-foreground focus:ring-2 focus:ring-primary focus:border-transparent"
              >
                <option value="">Choose a team member...</option>
                {teamMembers.map((member) => (
                  <option key={member.id} value={member.id}>
                    {member.name} ({member.role}) - {member.current_tasks_count} current tasks
                  </option>
                ))}
              </select>
            </div>
          </div>
        </div>
      </div>

      {/* Task Details Preview */}
      {selectedTask && (
        <div className="bg-muted p-4 rounded-md">
          <h4 className="font-medium text-foreground mb-3 flex items-center">
            <ClipboardList className="w-4 h-4 mr-2" />
            Task Details
          </h4>
          <div className="space-y-2 text-sm">
            <p><strong>Name:</strong> {selectedTask.name}</p>
            {selectedTask.description && (
              <p><strong>Description:</strong> {selectedTask.description}</p>
            )}
            <p><strong>Project:</strong> {selectedTask.project_name}</p>
            <p><strong>Due Date:</strong> {formatDate(selectedTask.due_date)}</p>
            <p>
              <strong>Priority:</strong>{' '}
              <span className={`capitalize ${getPriorityColor(selectedTask.priority)}`}>
                {selectedTask.priority}
              </span>
            </p>
            <p><strong>Status:</strong> <span className="capitalize">{selectedTask.status}</span></p>
          </div>
        </div>
      )}

      {/* Member Workload Preview */}
      {selectedMember && (
        <div className="bg-muted p-4 rounded-md">
          <h4 className="font-medium text-foreground mb-3 flex items-center">
            <User className="w-4 h-4 mr-2" />
            Member Workload
          </h4>
          <div className="space-y-2 text-sm">
            <p><strong>Name:</strong> {selectedMember.name}</p>
            <p><strong>Role:</strong> {selectedMember.role}</p>
            <p><strong>Email:</strong> {selectedMember.email}</p>
            <p>
              <strong>Current Tasks:</strong>{' '}
              <span className={selectedMember.current_tasks_count > 3 ? 'text-red-500' : 'text-green-500'}>
                {selectedMember.current_tasks_count} tasks
              </span>
            </p>
            {selectedMember.current_tasks_count > 5 && (
              <p className="text-red-500 text-xs flex items-center">
                <AlertCircle className="w-3 h-3 mr-1" />
                This member has a heavy workload
              </p>
            )}
          </div>
        </div>
      )}

      {/* Assignment Summary */}
      {selectedTask && selectedMember && (
        <div className="bg-primary/10 p-4 rounded-md border border-primary/20">
          <h4 className="font-medium text-foreground mb-2 flex items-center">
            <TrendingUp className="w-4 h-4 mr-2" />
            Assignment Summary
          </h4>
          <p className="text-sm text-foreground">
            Assigning <strong>"{selectedTask.name}"</strong> to{' '}
            <strong>{selectedMember.name}</strong>
          </p>
          {selectedMember.current_tasks_count > 0 && (
            <p className="text-xs text-muted-foreground mt-1">
              This member will have {selectedMember.current_tasks_count + 1} tasks in progress
            </p>
          )}
        </div>
      )}

      <div className="flex space-x-3 pt-4">
        <Button
          type="button"
          variant="outline"
          onClick={onCancel}
          disabled={isLoading}
          className="flex-1"
        >
          Cancel
        </Button>
        <Button
          type="submit"
          disabled={isLoading || !formData.task_id || !formData.assigned_to}
          className="flex-1 bg-primary hover:bg-primary/90"
        >
          {isLoading ? (
            <>
              <Loader2 className="w-4 h-4 animate-spin mr-2" />
              Assigning...
            </>
          ) : (
            'Assign Task'
          )}
        </Button>
      </div>
    </form>
  );
}