import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { supabase } from '@/components/supabaseClient';
import { Loader2 } from 'lucide-react';

interface AssignProjectFormProps {
  onSuccess: () => void;
  onCancel: () => void;
}

interface Project {
  id: string;
  name: string;
  description: string;
  status: string;
  team_id: string | null;
}

interface Team {
  id: string;
  name: string;
  description: string;
}

export function AssignProjectForm({ onSuccess, onCancel }: AssignProjectFormProps) {
  const [formData, setFormData] = useState({
    project_id: '',
    team_id: ''
  });
  const [projects, setProjects] = useState<Project[]>([]);
  const [teams, setTeams] = useState<Team[]>([]);
  const [isLoading, setIsLoading] = useState(false);
  const [isFetching, setIsFetching] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    fetchData();
  }, []);

  const fetchData = async () => {
    try {
      setIsFetching(true);
      setError(null);

      // Fetch projects that are not assigned to any team or all projects
      const { data: projectsData, error: projectsError } = await supabase
        .from('projects')
        .select('*')
        .order('name');

      // Fetch all teams
      const { data: teamsData, error: teamsError } = await supabase
        .from('teams')
        .select('*')
        .order('name');

      if (projectsError) throw projectsError;
      if (teamsError) throw teamsError;

      setProjects(projectsData || []);
      setTeams(teamsData || []);
    } catch (err) {
      console.error('Error fetching data:', err);
      setError('Failed to load data. Please try again.');
    } finally {
      setIsFetching(false);
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.project_id || !formData.team_id) {
      setError('Please select both a project and a team');
      return;
    }

    setIsLoading(true);
    setError(null);

    try {
      const { error: updateError } = await supabase
        .from('projects')
        .update({ 
          team_id: formData.team_id,
          updated_at: new Date().toISOString()
        })
        .eq('id', formData.project_id);

      if (updateError) throw updateError;

      // Show success message
      const projectName = projects.find(p => p.id === formData.project_id)?.name;
      const teamName = teams.find(t => t.id === formData.team_id)?.name;
      
      console.log(`Successfully assigned ${projectName} to ${teamName}`);
      onSuccess();
      
    } catch (err) {
      console.error('Error assigning project:', err);
      setError('Failed to assign project. Please try again.');
    } finally {
      setIsLoading(false);
    }
  };

  const handleProjectChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    setFormData({ ...formData, project_id: e.target.value });
    setError(null);
  };

  const handleTeamChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    setFormData({ ...formData, team_id: e.target.value });
    setError(null);
  };

  if (isFetching) {
    return (
      <div className="flex flex-col items-center justify-center py-8">
        <Loader2 className="w-8 h-8 animate-spin text-primary mb-4" />
        <p className="text-muted-foreground">Loading projects and teams...</p>
      </div>
    );
  }

  return (
    <form onSubmit={handleSubmit} className="space-y-6">
      <div>
        <h3 className="text-lg font-medium text-foreground mb-4">
          Assign Project to Team
        </h3>
        
        {error && (
          <div className="bg-destructive/15 text-destructive p-3 rounded-md mb-4">
            <p className="text-sm">{error}</p>
          </div>
        )}

        <div className="space-y-4">
          <div>
            <label className="block text-sm font-medium text-foreground mb-2">
              Select Project
            </label>
            <select
              required
              value={formData.project_id}
              onChange={handleProjectChange}
              className="w-full p-3 border border-border rounded-md bg-background text-foreground focus:ring-2 focus:ring-primary focus:border-transparent"
            >
              <option value="">Choose a project...</option>
              {projects.map((project) => (
                <option key={project.id} value={project.id}>
                  {project.name} {project.team_id ? '(Already assigned)' : ''}
                </option>
              ))}
            </select>
            <p className="text-xs text-muted-foreground mt-1">
              Select the project you want to assign to a team
            </p>
          </div>

          <div>
            <label className="block text-sm font-medium text-foreground mb-2">
              Assign to Team
            </label>
            <select
              required
              value={formData.team_id}
              onChange={handleTeamChange}
              className="w-full p-3 border border-border rounded-md bg-background text-foreground focus:ring-2 focus:ring-primary focus:border-transparent"
            >
              <option value="">Choose a team...</option>
              {teams.map((team) => (
                <option key={team.id} value={team.id}>
                  {team.name}
                </option>
              ))}
            </select>
            <p className="text-xs text-muted-foreground mt-1">
              Select the team that will work on this project
            </p>
          </div>
        </div>
      </div>

      {formData.project_id && formData.team_id && (
        <div className="bg-muted p-4 rounded-md">
          <h4 className="font-medium text-foreground mb-2">Assignment Summary</h4>
          <p className="text-sm text-muted-foreground">
            You are assigning{' '}
            <strong>{projects.find(p => p.id === formData.project_id)?.name}</strong>{' '}
            to{' '}
            <strong>{teams.find(t => t.id === formData.team_id)?.name}</strong>
          </p>
        </div>
      )}

      <div className="flex space-x-3 pt-4">
        <Button
          type="button"
          variant="outline"
          onClick={onCancel}
          disabled={isLoading}
          className="flex-1"
        >
          Cancel
        </Button>
        <Button
          type="submit"
          disabled={isLoading || !formData.project_id || !formData.team_id}
          className="flex-1 bg-primary hover:bg-primary/90"
        >
          {isLoading ? (
            <>
              <Loader2 className="w-4 h-4 animate-spin mr-2" />
              Assigning...
            </>
          ) : (
            'Assign Project'
          )}
        </Button>
      </div>
    </form>
  );
}