import React, { useState, useEffect, useCallback } from 'react';
import { Button } from '@/components/ui/button';
import { supabase } from '@/components/supabaseClient';
import { Loader2, AlertCircle, CheckCircle, XCircle, Users } from 'lucide-react';

interface AddTeamMemberFormProps {
  onSuccess: () => void;
  onCancel: () => void;
}

interface Team {
  id: string;
  name: string;
  description: string;
}

interface OrganizationMember {
  id: string;
  name: string;
  email: string;
  role: string;
  status: string;
  is_assigned: boolean;
  current_team?: string;
}

export function AddTeamMemberForm({ onSuccess, onCancel }: AddTeamMemberFormProps) {
  const [formData, setFormData] = useState({
    member_id: '',
    team_id: ''
  });
  const [teams, setTeams] = useState<Team[]>([]);
  const [organizationMembers, setOrganizationMembers] = useState<OrganizationMember[]>([]);
  const [isLoading, setIsLoading] = useState(false);
  const [isFetching, setIsFetching] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [successMessage, setSuccessMessage] = useState<string | null>(null);

  useEffect(() => {
    fetchData();
  }, []);

  const fetchData = async () => {
    try {
      setIsFetching(true);
      setError(null);

      // Fetch all teams
      const { data: teamsData, error: teamsError } = await supabase
        .from('teams')
        .select('*')
        .order('name');

      // Fetch organization members
      const { data: membersData, error: membersError } = await supabase
        .from('organization_members')
        .select('*')
        .order('name');

      if (teamsError) throw teamsError;
      if (membersError) throw membersError;

      setTeams(teamsData || []);

      // Check which members are already assigned to teams
      const { data: existingTeamMembers, error: teamMembersError } = await supabase
        .from('team_members')
        .select('organization_member_id, teams(name)');

      if (teamMembersError) throw teamMembersError;

      // Map existing assignments
      const assignedMembers = new Map();
      existingTeamMembers?.forEach(member => {
        assignedMembers.set(member.organization_member_id, member.teams?.name);
      });

      // Enrich organization members with assignment status
      const enrichedMembers = (membersData || []).map(member => ({
        ...member,
        is_assigned: assignedMembers.has(member.id),
        current_team: assignedMembers.get(member.id) || null
      }));

      setOrganizationMembers(enrichedMembers);

    } catch (err) {
      console.error('Error fetching data:', err);
      setError('Failed to load data. Please try again.');
    } finally {
      setIsFetching(false);
    }
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
    setError(null);
    setSuccessMessage(null);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    // Clear previous messages
    setError(null);
    setSuccessMessage(null);

    // Validate form
    if (!formData.member_id || !formData.team_id) {
      setError('Please select both a team member and a team');
      return;
    }

    setIsLoading(true);

    try {
      // Get the selected member details
      const selectedMember = organizationMembers.find(m => m.id === formData.member_id);
      
      if (!selectedMember) {
        setError('Selected member not found');
        return;
      }

      if (selectedMember.is_assigned) {
        setError(`This member is already assigned to team: ${selectedMember.current_team}`);
        return;
      }

      // Add member to team_members table
      const { error: insertError } = await supabase
        .from('team_members')
        .insert([{
          organization_member_id: formData.member_id,
          name: selectedMember.name,
          email: selectedMember.email,
          role: selectedMember.role,
          team_id: formData.team_id,
          created_at: new Date().toISOString()
        }]);

      if (insertError) {
        if (insertError.code === '23505') {
          setError('This member is already assigned to a team');
        } else {
          throw insertError;
        }
        return;
      }

      setSuccessMessage(`Successfully added ${selectedMember.name} to the team!`);
      
      // Reset form
      setFormData({
        member_id: '',
        team_id: ''
      });

      // Refresh data to update assignment status
      await fetchData();
      
      // Call success callback after a delay
      setTimeout(() => {
        onSuccess();
      }, 1500);

    } catch (error: any) {
      console.error('Error adding team member:', error);
      setError('Failed to add team member. Please try again.');
    } finally {
      setIsLoading(false);
    }
  };

  if (isFetching) {
    return (
      <div className="flex flex-col items-center justify-center py-8">
        <Loader2 className="w-8 h-8 animate-spin text-primary mb-4" />
        <p className="text-muted-foreground">Loading teams and members...</p>
      </div>
    );
  }

  const availableMembers = organizationMembers.filter(member => !member.is_assigned);
  const assignedMembers = organizationMembers.filter(member => member.is_assigned);

  return (
    <form onSubmit={handleSubmit} className="space-y-6">
      <div>
        <h3 className="text-lg font-medium text-foreground mb-4">Add Team Member from Organization</h3>
        
        {error && (
          <div className="bg-destructive/15 text-destructive p-3 rounded-md mb-4 flex items-center">
            <AlertCircle className="w-4 h-4 mr-2" />
            <p className="text-sm">{error}</p>
          </div>
        )}

        {successMessage && (
          <div className="bg-green-50 border border-green-200 text-green-800 p-3 rounded-md mb-4 flex items-center">
            <CheckCircle className="w-4 h-4 mr-2" />
            <p className="text-sm">{successMessage}</p>
          </div>
        )}

        <div className="space-y-4">
          {/* Team Member Selection */}
          <div>
            <label className="block text-sm font-medium text-foreground mb-2">
              Select Team Member *
            </label>
            <div className="relative">
              <Users className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-muted-foreground" />
              <select
                name="member_id"
                required
                value={formData.member_id}
                onChange={handleInputChange}
                className="w-full pl-10 p-3 border border-border rounded-md bg-background text-foreground focus:ring-2 focus:ring-primary focus:border-transparent"
              >
                <option value="">Choose a team member...</option>
                <optgroup label="Available Members">
                  {availableMembers.map((member) => (
                    <option key={member.id} value={member.id}>
                      {member.name} - {member.role} ({member.email})
                    </option>
                  ))}
                </optgroup>
                {assignedMembers.length > 0 && (
                  <optgroup label="Already Assigned">
                    {assignedMembers.map((member) => (
                      <option key={member.id} value={member.id} disabled>
                        {member.name} - {member.role} (Assigned to: {member.current_team})
                      </option>
                    ))}
                  </optgroup>
                )}
              </select>
            </div>
            <p className="text-xs text-muted-foreground mt-1">
              {availableMembers.length} available members, {assignedMembers.length} already assigned to teams
            </p>
          </div>

          {/* Team Selection */}
          <div>
            <label className="block text-sm font-medium text-foreground mb-2">
              Assign to Team *
            </label>
            <select
              name="team_id"
              required
              value={formData.team_id}
              onChange={handleInputChange}
              className="w-full p-3 border border-border rounded-md bg-background text-foreground focus:ring-2 focus:ring-primary focus:border-transparent"
            >
              <option value="">Choose a team...</option>
              {teams.map((team) => (
                <option key={team.id} value={team.id}>
                  {team.name}
                </option>
              ))}
            </select>
          </div>

          {/* Selected Member Preview */}
          {formData.member_id && (
            <div className="bg-muted p-4 rounded-md">
              <h4 className="font-medium text-foreground mb-2">Member Details</h4>
              {(() => {
                const member = organizationMembers.find(m => m.id === formData.member_id);
                return member ? (
                  <div className="text-sm space-y-1">
                    <p><strong>Name:</strong> {member.name}</p>
                    <p><strong>Email:</strong> {member.email}</p>
                    <p><strong>Role:</strong> {member.role}</p>
                    <p><strong>Status:</strong> {member.status}</p>
                    {member.is_assigned && (
                      <p className="text-yellow-600">
                        <strong>Note:</strong> Already assigned to {member.current_team}
                      </p>
                    )}
                  </div>
                ) : null;
              })()}
            </div>
          )}
        </div>
      </div>

      <div className="flex space-x-3 pt-4">
        <Button type="button" variant="outline" onClick={onCancel} disabled={isLoading}>
          Cancel
        </Button>
        <Button 
          type="submit" 
          disabled={isLoading || !formData.member_id || !formData.team_id}
          className="bg-primary hover:bg-primary/90"
        >
          {isLoading ? (
            <>
              <Loader2 className="w-4 h-4 animate-spin mr-2" />
              Adding to Team...
            </>
          ) : (
            'Add to Team'
          )}
        </Button>
      </div>
    </form>
  );
}