import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { supabase } from '@/components/supabaseClient';
import { Button } from '@/components/ui/button';
import { Card, CardHeader, CardTitle, CardDescription, CardContent, CardFooter } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { CheckCircle2, ChevronRight, Zap, Sparkles, Shield } from 'lucide-react';
import { usePaystackPayment } from 'react-paystack';

// Paystack public key (store this in your environment configuration)
const PAYSTACK_PUBLIC_KEY = import.meta.env.VITE_PAYSTACK_PUBLIC_KEY || 'pk_test_your_public_key';

const ProUpgrade = () => {
  const [loading, setLoading] = useState(false);
  const [selectedPlan, setSelectedPlan] = useState('pro');
  const [userEmail, setUserEmail] = useState('');
  const [currentPlan, setCurrentPlan] = useState('basic');
  const [userDetails, setUserDetails] = useState<any>(null);
  const navigate = useNavigate();

  // Fetch user data on component mount
  useEffect(() => {
    const fetchUserData = async () => {
      const { data: { user }, error } = await supabase.auth.getUser();
      if (user) {
        setUserEmail(user.email || '');
        setUserDetails({
          email: user.email,
          name: user.user_metadata?.full_name || '',
        });

        // Check current subscription
        const { data: subscription, error: subError } = await supabase
          .from('subscriptions')
          .select('plan')
          .eq('user_id', user.id)
          .single();

        if (!subError && subscription) {
          setCurrentPlan(subscription.plan);
        }
      }
    };

    fetchUserData();
  }, []);

  const plans = [
    {
      id: 'pro',
      name: 'Pro',
      price: 2900, // in kobo (₦29.00)
      displayPrice: '$9',
      interval: 'month',
      description: 'For serious individuals',
      features: [
        'All Basic features',
        'Innovation Tracker',
        'Advanced analytics',
        'Priority support',
        'Custom integrations'
      ],
      paystack_plan: 'PLN_xxxxxx' // Replace with your actual Paystack plan code
    },
    {
      id: 'enterprise',
      name: 'Enterprise',
      price: 0, // Custom pricing
      displayPrice: '$29',
      interval: '',
      description: 'For organizations and large teams',
      features: [
        'All Pro features',
        'Organization-wide tools',
        'Team dashboards',
        'Remote Management',
        'KPIs and OKRs tracking',
        'Organizational Health Metrics',
      ],
      paystack_plan: null
    },
    {
      id: 'civic',
      name: 'Civic',
      price: 0, // Custom pricing
      displayPrice: '$49',
      interval: '',
      description: 'For Politicians and Leaders',
      features: [
        'All Pro features',
        'Tools for Leadership development',
        'Strategic Planning',
        'Public engagement',
       
      ],
      paystack_plan: null
    }
  ];

  // Paystack config
  const config = {
    reference: (new Date()).getTime().toString(),
    email: userEmail,
    amount: plans.find(p => p.id === selectedPlan)?.price || 0,
    publicKey: PAYSTACK_PUBLIC_KEY || 'your_paystack_public_key',
    currency: 'KSH',
    plan: plans.find(p => p.id === selectedPlan)?.paystack_plan || '',
    metadata: {
      plan: selectedPlan,
      userId: userDetails?.id || ''
    },
    channels: ['card', 'bank', 'ussd', 'qr', 'mobile_money'],
    label: "Upgrade to Pro Plan",
  };

  const initializePayment = usePaystackPayment(config);

  const onSuccess = async (response: any) => {
    try {
      // Verify transaction with your backend
      const verifyResponse = await fetch('/api/verify-payment', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          reference: response.reference,
          userId: userDetails?.id
        }),
      });

      const verification = await verifyResponse.json();

      if (verification.success) {
        // Update subscription in Supabase
        const { error } = await supabase
          .from('subscriptions')
          .upsert({
            user_id: userDetails?.id,
            plan: selectedPlan,
            status: 'active',
            payment_reference: response.reference,
            starts_at: new Date().toISOString(),
            ends_at: new Date(new Date().setMonth(new Date().getMonth() + 1)).toISOString()
          });

        if (error) throw error;

        toast.success('Payment successful! Your account has been upgraded.');
        navigate('/upgrade-success');
      } else {
        throw new Error('Payment verification failed');
      }
    } catch (error) {
      console.error('Payment processing error:', error);
      toast.error('Payment verification failed. Please contact support.');
    }
  };

  const onClose = () => {
    toast('Payment window closed', { icon: 'ℹ️' });
  };

  const handleUpgrade = async () => {
  setLoading(true);
  try {
    // First get payment details from your secure endpoint
    const { data, error } = await supabase.functions.invoke('initialize-payment', {
      body: { plan: selectedPlan, email: userEmail, userId: userDetails?.id }
    });

    if (error) throw error;

    // Then initialize Paystack with the returned details
    initializePayment({
      ...config,
      publicKey: data.publicKey,
      reference: data.reference,
      amount: data.amount
    }, onSuccess, onClose);
  } catch (error) {
    toast.error('Failed to initialize payment');
  } finally {
    setLoading(false);
  }
};

  // Edge case: User is already on this plan
  if (currentPlan === selectedPlan) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <Card className="max-w-md text-center">
          <CardHeader>
            <CardTitle>You're already on this plan!</CardTitle>
          </CardHeader>
          <CardContent>
            <p className="text-muted-foreground mb-4">
              You're currently subscribed to the {currentPlan} plan.
            </p>
            <Button onClick={() => navigate('/dashboard')}>
              Return to Dashboard
            </Button>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-b from-background to-muted/20 py-12 px-4 sm:px-6 lg:px-8">
      <div className="max-w-4xl mx-auto">
        <div className="text-center mb-12">
          <h1 className="text-4xl font-bold tracking-tight mb-4">
            {currentPlan === 'basic' ? 'Upgrade Your Plan' : 'Change Your Plan'}
          </h1>
          <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
            Unlock powerful features to accelerate your development and growth
          </p>
        </div>

        <div className="grid md:grid-cols-3 gap-8 mb-10">
          {plans.map((plan) => (
            <Card 
              key={plan.id}
              className={`relative overflow-hidden transition-all hover:shadow-lg ${
                selectedPlan === plan.id ? 'ring-2 ring-primary' : ''
              }`}
            >
              {selectedPlan === plan.id && (
                <div className="absolute top-0 right-0 bg-primary text-primary-foreground px-3 py-1 text-xs font-bold rounded-bl-lg">
                  SELECTED
                </div>
              )}
              <CardHeader>
                <div className="flex justify-between items-start">
                  <div>
                    <CardTitle className="text-2xl flex items-center gap-2">
                      {plan.name}
                      {plan.id === 'pro' && <Zap className="w-5 h-5 text-yellow-500" />}
                      {plan.id === 'enterprise' && <Sparkles className="w-5 h-5 text-purple-500" />}
                      {plan.id === 'civic' && <Sparkles className="w-5 h-5 text-purple-500" />}
                    </CardTitle>
                    <CardDescription>{plan.description}</CardDescription>
                  </div>
                  <div className="text-right">
                    <div className="text-3xl font-bold">{plan.displayPrice}</div>
                    <div className="text-sm text-muted-foreground">{plan.interval}</div>
                  </div>
                </div>
              </CardHeader>
              <CardContent>
                <ul className="space-y-3">
                  {plan.features.map((feature, i) => (
                    <li key={i} className="flex items-center">
                      <CheckCircle2 className="w-5 h-5 text-green-500 mr-2" />
                      <span>{feature}</span>
                    </li>
                  ))}
                </ul>
              </CardContent>
              <CardFooter>
                <Button
                  variant={selectedPlan === plan.id ? 'default' : 'outline'}
                  className="w-full"
                  onClick={() => setSelectedPlan(plan.id)}
                >
                  {selectedPlan === plan.id ? (
                    <span>Selected</span>
                  ) : (
                    <span>{plan.id === 'enterprise' ? 'Select Plan' : 'Select Plan'}</span>
                  )}
                </Button>
              </CardFooter>
            </Card>
          ))}
        </div>

        <div className="bg-card border rounded-xl p-6 mb-8">
          <h3 className="text-lg font-semibold mb-4 flex items-center">
            <Shield className="w-5 h-5 text-blue-500 mr-2" />
            Payment Security
          </h3>
          <p className="text-muted-foreground mb-4">
            All payments are securely processed using Paystack. Your financial information is never stored on our servers.
          </p>
          <div className="flex flex-wrap gap-4">
            <Badge variant="secondary">PCI DSS Compliant</Badge>
            <Badge variant="secondary">Bank-level Security</Badge>
            <Badge variant="secondary">Secure Payments</Badge>
          </div>
        </div>

        <div className="flex flex-col sm:flex-row justify-between items-center gap-4">
          <Button 
            variant="ghost" 
            onClick={() => navigate(-1)}
          >
            Back
          </Button>
          <Button
            size="lg"
            className="w-full sm:w-auto"
            onClick={handleUpgrade}
            disabled={loading || !userEmail}
          >
            {loading ? (
              'Processing...'
            ) : (
              <>
                {selectedPlan === 'enterprise' ? 'Contact Sales' : 'Complete Payment'} 
                <ChevronRight className="w-4 h-4 ml-2" />
              </>
            )}
          </Button>
        </div>
      </div>
    </div>
  );
};

export default ProUpgrade;