// src/components/OrganizationDashboard/index.tsx
import { useState, useEffect } from 'react';
import { useParams, useNavigate, Outlet } from 'react-router-dom';
import { supabase } from '../../supabaseClient';
import { Organization } from '../../types/organization';
import { Sidebar } from './Sidebar';
import { Header } from './Header';
import { Loader } from '../../components/ui/Loader';

export default function OrganizationDashboard() {
  const { orgId } = useParams();
  const navigate = useNavigate();
  const [organization, setOrganization] = useState<Organization | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const fetchOrganization = async () => {
      try {
        setLoading(true);
        
        // Fetch organization data
        const { data, error: orgError } = await supabase
          .from('organizations')
          .select('*')
          .eq('id', orgId)
          .single();

        if (orgError) throw orgError;

        // Fetch members
        const { data: membersData, error: membersError } = await supabase
          .from('organization_members')
          .select('*')
          .eq('organization_id', orgId);

        if (membersError) throw membersError;

        setOrganization({
          ...data,
          members: membersData.map(member => ({
            id: member.user_id,
            name: member.user_name,
            email: member.user_email,
            role: member.role,
            joinedAt: new Date(member.joined_at),
            status: member.status
          }))
        });
      } catch (err) {
        setError(err.message || 'Failed to load organization data');
        navigate('/');
      } finally {
        setLoading(false);
      }
    };

    fetchOrganization();
  }, [orgId, navigate]);

  if (loading) return <Loader />;
  if (error) return <div className="error-message">{error}</div>;
  if (!organization) return null;

  return (
    <div className="organization-dashboard">
      <Header organization={organization} />
      
      <div className="dashboard-content">
        <Sidebar organization={organization} />
        
        <main className="main-content">
          <Outlet context={{ organization }} />
        </main>
      </div>
    </div>
  );
}