// src/components/OrganizationDashboard/MembersList.tsx
import { useState } from 'react';
import { Organization, OrganizationMember } from '../../types/organization';
import { FaUser, FaTrash, FaEdit, FaEnvelope } from 'react-icons/fa';
import { Button } from '../../components/ui/Button';
import { InviteMembersModal } from './InviteMembers';

interface MembersListProps {
  organization: Organization;
}

export function MembersList({ organization }: MembersListProps) {
  const [showInviteModal, setShowInviteModal] = useState(false);
  const [filter, setFilter] = useState<'all' | 'active' | 'pending'>('all');

  const filteredMembers = organization.members.filter(member => {
    if (filter === 'all') return true;
    return member.status === filter;
  });

  return (
    <div className="members-list">
      <div className="members-header">
        <h2>Organization Members</h2>
        
        <div className="members-actions">
          <select 
            value={filter} 
            onChange={(e) => setFilter(e.target.value as any)}
            className="filter-select"
          >
            <option value="all">All Members</option>
            <option value="active">Active</option>
            <option value="pending">Pending</option>
          </select>
          
          <Button onClick={() => setShowInviteModal(true)}>
            <FaEnvelope /> Invite Members
          </Button>
        </div>
      </div>
      
      <table className="members-table">
        <thead>
          <tr>
            <th>Name</th>
            <th>Email</th>
            <th>Role</th>
            <th>Status</th>
            <th>Actions</th>
          </tr>
        </thead>
        <tbody>
          {filteredMembers.map(member => (
            <tr key={member.id}>
              <td>
                <FaUser /> {member.name}
              </td>
              <td>{member.email}</td>
              <td>
                <span className={`role-badge ${member.role}`}>
                  {member.role}
                </span>
              </td>
              <td>
                <span className={`status-badge ${member.status}`}>
                  {member.status}
                </span>
              </td>
              <td className="actions">
                <button className="edit-btn">
                  <FaEdit />
                </button>
                <button className="delete-btn">
                  <FaTrash />
                </button>
              </td>
            </tr>
          ))}
        </tbody>
      </table>
      
      {showInviteModal && (
        <InviteMembersModal 
          organization={organization}
          onClose={() => setShowInviteModal(false)}
        />
      )}
    </div>
  );
}