import { useState, useEffect } from "react";
import { supabase } from "@/components/supabaseClient";
import {
  Card,
  CardContent,
  CardHeader,
  CardTitle,
  CardDescription,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Checkbox } from "@/components/ui/checkbox";
import { Plus, Trash2, Calendar, TrendingUp, Target, Edit3 } from "lucide-react";
import { useNavigate } from "react-router-dom";
import { format, startOfWeek, endOfWeek, eachDayOfInterval, isToday, isSameDay, parseISO } from "date-fns";

interface Habit {
  id: string;
  name: string;
  description?: string;
  user_id: string;
  created_at: string;
  target_days?: number[];
  color?: string;
}

interface HabitCompletion {
  id: string;
  habit_id: string;
  user_id: string;
  completed_date: string;
  created_at: string;
}

export const HabitTracker = () => {
  const navigate = useNavigate();
  const [userId, setUserId] = useState<string | null>(null);
  const [habits, setHabits] = useState<Habit[]>([]);
  const [completions, setCompletions] = useState<HabitCompletion[]>([]);
  const [newHabitName, setNewHabitName] = useState("");
  const [newHabitDescription, setNewHabitDescription] = useState("");
  const [isLoading, setIsLoading] = useState(true);
  const [selectedDate, setSelectedDate] = useState(new Date());
  const [editingHabit, setEditingHabit] = useState<Habit | null>(null);
  const [showEditModal, setShowEditModal] = useState(false);

  // Get the current week dates
  const weekDates = eachDayOfInterval({
    start: startOfWeek(selectedDate, { weekStartsOn: 1 }), // Monday start
    end: endOfWeek(selectedDate, { weekStartsOn: 1 }),
  });

  useEffect(() => {
    const fetchUserAndHabits = async () => {
      setIsLoading(true);
      try {
        // Get current user
        const { data: { user }, error: userError } = await supabase.auth.getUser();
        if (userError) throw userError;
        setUserId(user?.id || null);

        if (user?.id) {
          // Fetch habits
          const { data: habitsData, error: habitsError } = await supabase
            .from("habits")
            .select("*")
            .eq("user_id", user.id)
            .order("created_at", { ascending: true });

          if (habitsError) throw habitsError;

          // Fetch completions for the current week
          const weekStart = startOfWeek(selectedDate, { weekStartsOn: 1 });
          const weekEnd = endOfWeek(selectedDate, { weekStartsOn: 1 });

          const { data: completionsData, error: completionsError } = await supabase
            .from("habit_completions")
            .select("*")
            .eq("user_id", user.id)
            .gte("completed_date", weekStart.toISOString())
            .lte("completed_date", weekEnd.toISOString());

          if (completionsError) throw completionsError;

          setHabits(habitsData || []);
          setCompletions(completionsData || []);
        }
      } catch (error) {
        console.error("Error fetching data:", error);
      } finally {
        setIsLoading(false);
      }
    };

    fetchUserAndHabits();
  }, [selectedDate]);

  const handleCreateHabit = async () => {
    if (!userId || !newHabitName.trim()) return;

    try {
      const { data, error } = await supabase
        .from("habits")
        .insert([
          {
            user_id: userId,
            name: newHabitName.trim(),
            description: newHabitDescription.trim(),
            color: `hsl(${Math.floor(Math.random() * 360)}, 70%, 60%)`,
          },
        ])
        .select();

      if (error) throw error;

      setHabits([...habits, data[0]]);
      setNewHabitName("");
      setNewHabitDescription("");
    } catch (error) {
      console.error("Error creating habit:", error);
      alert("Failed to create habit");
    }
  };

  const handleDeleteHabit = async (habitId: string) => {
    try {
      const { error } = await supabase
        .from("habits")
        .delete()
        .eq("id", habitId);

      if (error) throw error;

      // Also delete related completions
      await supabase
        .from("habit_completions")
        .delete()
        .eq("habit_id", habitId);

      setHabits(habits.filter(habit => habit.id !== habitId));
      setCompletions(completions.filter(completion => completion.habit_id !== habitId));
    } catch (error) {
      console.error("Error deleting habit:", error);
      alert("Failed to delete habit");
    }
  };

  const handleUpdateHabit = async (updatedHabit: Habit) => {
    try {
      const { error } = await supabase
        .from("habits")
        .update({
          name: updatedHabit.name,
          description: updatedHabit.description,
          target_days: updatedHabit.target_days,
        })
        .eq("id", updatedHabit.id);

      if (error) throw error;

      setHabits(habits.map(habit => 
        habit.id === updatedHabit.id ? updatedHabit : habit
      ));
      setShowEditModal(false);
      setEditingHabit(null);
    } catch (error) {
      console.error("Error updating habit:", error);
      alert("Failed to update habit");
    }
  };

  const toggleHabitCompletion = async (habitId: string, date: Date) => {
    if (!userId) return;

    const dateString = date.toISOString().split('T')[0];
    const existingCompletion = completions.find(
      comp => comp.habit_id === habitId && comp.completed_date === dateString
    );

    try {
      if (existingCompletion) {
        // Remove completion
        const { error } = await supabase
          .from("habit_completions")
          .delete()
          .eq("id", existingCompletion.id);

        if (error) throw error;

        setCompletions(completions.filter(comp => comp.id !== existingCompletion.id));
      } else {
        // Add completion
        const { data, error } = await supabase
          .from("habit_completions")
          .insert([
            {
              user_id: userId,
              habit_id: habitId,
              completed_date: dateString,
            },
          ])
          .select();

        if (error) throw error;

        setCompletions([...completions, data[0]]);
      }
    } catch (error) {
      console.error("Error toggling habit completion:", error);
    }
  };

  const isHabitCompleted = (habitId: string, date: Date) => {
    const dateString = date.toISOString().split('T')[0];
    return completions.some(
      comp => comp.habit_id === habitId && comp.completed_date === dateString
    );
  };

  const getCompletionRate = (habitId: string) => {
    const habitCompletions = completions.filter(comp => comp.habit_id === habitId);
    return (habitCompletions.length / 7) * 100; // Weekly rate
  };

  const navigateToPreviousWeek = () => {
    setSelectedDate(prev => {
      const newDate = new Date(prev);
      newDate.setDate(newDate.getDate() - 7);
      return newDate;
    });
  };

  const navigateToNextWeek = () => {
    setSelectedDate(prev => {
      const newDate = new Date(prev);
      newDate.setDate(newDate.getDate() + 7);
      return newDate;
    });
  };

  const navigateToCurrentWeek = () => {
    setSelectedDate(new Date());
  };

  if (isLoading) {
    return (
      <div className="flex justify-center items-center h-screen">
        <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-indigo-500"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6 max-w-6xl mx-auto pb-10 px-4">
      <Button
        variant="outline"
        onClick={() => navigate("/IndividualDashboard")}
        className="fixed top-4 left-4 z-50 bg-white shadow-md hover:bg-muted text-sm"
      >
        ← Back to Dashboard
      </Button>

      {/* Header */}
      <Card className="bg-gradient-to-r from-blue-500 to-purple-600 text-white border-0">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Target className="h-6 w-6" />
            Habit Tracker
          </CardTitle>
          <CardDescription className="text-blue-100">
            Build good habits one day at a time
          </CardDescription>
        </CardHeader>
      </Card>

      {/* Week Navigation */}
      <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
        <CardContent className="pt-6">
          <div className="flex items-center justify-between">
            <Button variant="outline" onClick={navigateToPreviousWeek}>
              ← Previous Week
            </Button>
            
            <div className="text-center">
              <h2 className="text-xl font-semibold">
                {format(startOfWeek(selectedDate, { weekStartsOn: 1 }), "MMM d")} -{" "}
                {format(endOfWeek(selectedDate, { weekStartsOn: 1 }), "MMM d, yyyy")}
              </h2>
              {!isToday(selectedDate) && (
                <Button variant="ghost" size="sm" onClick={navigateToCurrentWeek} className="mt-2">
                  <Calendar className="h-4 w-4 mr-2" />
                  Jump to Current Week
                </Button>
              )}
            </div>
            
            <Button variant="outline" onClick={navigateToNextWeek}>
              Next Week →
            </Button>
          </div>
        </CardContent>
      </Card>

      {/* Add New Habit */}
      <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Plus className="h-5 w-5" />
            Add New Habit
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="habit-name">Habit Name *</Label>
            <Input
              id="habit-name"
              placeholder="e.g., Morning Meditation, Exercise, Reading..."
              value={newHabitName}
              onChange={(e) => setNewHabitName(e.target.value)}
            />
          </div>
          <div className="space-y-2">
            <Label htmlFor="habit-description">Description (Optional)</Label>
            <Input
              id="habit-description"
              placeholder="Brief description of your habit..."
              value={newHabitDescription}
              onChange={(e) => setNewHabitDescription(e.target.value)}
            />
          </div>
          <Button 
            onClick={handleCreateHabit} 
            disabled={!newHabitName.trim()}
            className="w-full"
          >
            <Plus className="h-4 w-4 mr-2" />
            Add Habit
          </Button>
        </CardContent>
      </Card>

      {/* Habits List */}
      {habits.length === 0 ? (
        <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg text-center py-12">
          <CardContent>
            <Target className="h-12 w-12 mx-auto text-gray-400 mb-4" />
            <h3 className="text-lg font-semibold text-gray-600 mb-2">No habits yet</h3>
            <p className="text-gray-500">Add your first habit to start tracking!</p>
          </CardContent>
        </Card>
      ) : (
        <div className="space-y-4">
          {habits.map((habit) => (
            <Card key={habit.id} className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
              <CardHeader className="pb-3">
                <div className="flex items-center justify-between">
                  <CardTitle className="text-lg">{habit.name}</CardTitle>
                  <div className="flex items-center gap-2">
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => {
                        setEditingHabit(habit);
                        setShowEditModal(true);
                      }}
                    >
                      <Edit3 className="h-4 w-4" />
                    </Button>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => handleDeleteHabit(habit.id)}
                    >
                      <Trash2 className="h-4 w-4 text-red-500" />
                    </Button>
                  </div>
                </div>
                {habit.description && (
                  <CardDescription>{habit.description}</CardDescription>
                )}
                <div className="flex items-center gap-2 text-sm text-gray-600">
                  <TrendingUp className="h-4 w-4" />
                  <span>{Math.round(getCompletionRate(habit.id))}% completion this week</span>
                </div>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-7 gap-2">
                  {weekDates.map((date, index) => (
                    <div key={index} className="text-center">
                      <div className="text-xs font-medium text-gray-600 mb-1">
                        {format(date, "EEE").charAt(0)}
                      </div>
                      <div className="text-xs text-gray-500 mb-2">
                        {format(date, "d")}
                      </div>
                      <Checkbox
                        checked={isHabitCompleted(habit.id, date)}
                        onCheckedChange={() => toggleHabitCompletion(habit.id, date)}
                        className="h-8 w-8 data-[state=checked]:bg-green-500 data-[state=checked]:border-green-600"
                        disabled={date > new Date()}
                      />
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      )}

      {/* Edit Habit Modal */}
      {showEditModal && editingHabit && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <Card className="w-full max-w-md mx-4">
            <CardHeader>
              <CardTitle>Edit Habit</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <Label>Habit Name</Label>
                <Input
                  value={editingHabit.name}
                  onChange={(e) => setEditingHabit({
                    ...editingHabit,
                    name: e.target.value
                  })}
                />
              </div>
              <div className="space-y-2">
                <Label>Description</Label>
                <Input
                  value={editingHabit.description || ""}
                  onChange={(e) => setEditingHabit({
                    ...editingHabit,
                    description: e.target.value
                  })}
                />
              </div>
              <div className="flex gap-2">
                <Button
                  variant="outline"
                  onClick={() => setShowEditModal(false)}
                  className="flex-1"
                >
                  Cancel
                </Button>
                <Button
                  onClick={() => handleUpdateHabit(editingHabit)}
                  className="flex-1"
                >
                  Save Changes
                </Button>
              </div>
            </CardContent>
          </Card>
        </div>
      )}
    </div>
  );
};