import React, { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Progress } from '@/components/ui/progress';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { 
  ArrowLeft, 
  Users, 
  Target, 
  BarChart3, 
  FileText, 
  CheckCircle, 
  Clock, 
  TrendingUp,
  AlertCircle,
  Crown,
  Building,
  Gavel,
  Calendar,
  MessageSquare,
  Vote
} from 'lucide-react';
import { useAccessibility } from '@/hooks/useAccessibility';

interface ExecutiveClassProps {
  onBack: () => void;
}

const executiveModules = [
  {
    id: 'board-governance',
    title: 'Board Governance',
    description: 'Meeting management, decision tracking, and board member coordination',
    icon: Gavel,
    gradient: 'bg-gradient-primary',
    stats: {
      label: 'Active Boards',
      value: '3',
      trend: 'stable' as const
    }
  },
  {
    id: 'strategic-planning',
    title: 'Strategic Planning',
    description: 'Long-term goal setting, KPI tracking, and strategic initiative management',
    icon: Target,
    gradient: 'bg-gradient-hero',
    stats: {
      label: 'Strategic Goals',
      value: '8/12',
      trend: 'up' as const
    }
  },
  {
    id: 'parastatal-management',
    title: 'Parastatal Management',
    description: 'Oversight tools for government agencies and public sector organizations',
    icon: Building,
    gradient: 'bg-gradient-wellness',
    stats: {
      label: 'Organizations',
      value: '5 active',
      trend: 'up' as const
    }
  },
  {
    id: 'executive-reporting',
    title: 'Executive Reporting',
    description: 'Comprehensive reporting and analytics for leadership decision-making',
    icon: BarChart3,
    gradient: 'bg-gradient-growth',
    stats: {
      label: 'Reports',
      value: '12 monthly',
      trend: 'stable' as const
    }
  }
];

const executiveStats = [
  {
    title: 'Strategic Completion',
    value: '73%',
    description: 'Goals on track',
    icon: Target,
    color: 'text-primary'
  },
  {
    title: 'Board Decisions',
    value: '24',
    description: 'This quarter',
    icon: Vote,
    color: 'text-secondary-accent'
  },
  {
    title: 'Stakeholder Engagement',
    value: '92%',
    description: 'Satisfaction rate',
    icon: Users,
    color: 'text-accent-bright'
  },
  {
    title: 'Compliance Score',
    value: '98%',
    description: 'Regulatory adherence',
    icon: CheckCircle,
    color: 'text-green-500'
  }
];

const recentDecisions = [
  {
    id: '1',
    title: 'Budget Allocation for Q2 2025',
    status: 'approved',
    priority: 'high',
    date: '2025-01-10',
    followUp: '2025-01-25'
  },
  {
    id: '2',
    title: 'Strategic Partnership with Tech Consortium',
    status: 'pending',
    priority: 'medium',
    date: '2025-01-08',
    followUp: '2025-01-22'
  },
  {
    id: '3',
    title: 'Digital Transformation Initiative',
    status: 'in-progress',
    priority: 'high',
    date: '2025-01-05',
    followUp: '2025-02-01'
  }
];

const upcomingMeetings = [
  {
    id: '1',
    title: 'Board of Directors Meeting',
    date: '2025-01-20',
    time: '09:00 AM',
    attendees: 8,
    type: 'board'
  },
  {
    id: '2',
    title: 'Strategic Planning Session',
    date: '2025-01-22',
    time: '02:00 PM',
    attendees: 12,
    type: 'planning'
  },
  {
    id: '3',
    title: 'Stakeholder Review',
    date: '2025-01-25',
    time: '10:30 AM',
    attendees: 15,
    type: 'review'
  }
];

export function ExecutiveClass({ onBack }: ExecutiveClassProps) {
  const { announceToScreenReader } = useAccessibility();
  const [activeTab, setActiveTab] = useState('overview');

  const handleModuleClick = (moduleId: string) => {
    announceToScreenReader(`Navigating to ${moduleId} module`);
    // Handle navigation to specific executive tools
    console.log(`Navigate to ${moduleId}`);
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'approved': return 'text-green-500';
      case 'pending': return 'text-yellow-500';
      case 'in-progress': return 'text-blue-500';
      default: return 'text-muted-foreground';
    }
  };

  const getPriorityBadge = (priority: string) => {
    switch (priority) {
      case 'high': return <Badge variant="destructive">High</Badge>;
      case 'medium': return <Badge variant="secondary">Medium</Badge>;
      case 'low': return <Badge variant="outline">Low</Badge>;
      default: return null;
    }
  };

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <header className="border-b bg-card/50 backdrop-blur-sm sticky top-0 z-50">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-4">
              <Button 
                variant="ghost" 
                size="sm" 
                onClick={onBack}
                aria-label="Return to Elective Leadership Suite"
              >
                <ArrowLeft className="w-4 h-4 mr-2" />
                Back to Leadership Suite
              </Button>
              <div className="w-10 h-10 rounded-lg bg-gradient-primary flex items-center justify-center">
                <Crown className="w-6 h-6 text-white" />
              </div>
              <div>
                <h1 className="text-2xl font-bold bg-gradient-hero bg-clip-text text-transparent">
                  Executive Class
                </h1>
                <p className="text-sm text-muted-foreground">
                  Strategic leadership and governance tools
                </p>
              </div>
            </div>
            <Badge variant="default" className="bg-violet-800 text-white">
              Executive
            </Badge>
          </div>
        </div>
      </header>

      {/* Main Content */}
      <main className="container mx-auto px-6 py-8" tabIndex={-1}>
        <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
          <TabsList 
            className="grid w-full grid-cols-4 mb-8"
            role="tablist"
            aria-label="Executive dashboard sections"
          >
            <TabsTrigger value="overview" className="flex items-center">
              <BarChart3 className="w-4 h-4 mr-2" />
              Overview
            </TabsTrigger>
            <TabsTrigger value="decisions" className="flex items-center">
              <Vote className="w-4 h-4 mr-2" />
              Decisions
            </TabsTrigger>
            <TabsTrigger value="meetings" className="flex items-center">
              <Calendar className="w-4 h-4 mr-2" />
              Meetings
            </TabsTrigger>
            <TabsTrigger value="reports" className="flex items-center">
              <FileText className="w-4 h-4 mr-2" />
              Reports
            </TabsTrigger>
          </TabsList>

          <TabsContent value="overview" className="space-y-8">
            {/* Executive Stats */}
            <section aria-labelledby="stats-heading">
              <h2 id="stats-heading" className="text-lg font-semibold mb-4 text-foreground">
                Executive Dashboard
              </h2>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                {executiveStats.map((stat, index) => (
                  <Card key={stat.title} className="border-border/50">
                    <CardHeader className="pb-3">
                      <CardTitle className="text-base text-foreground flex items-center">
                        <stat.icon className={`w-5 h-5 mr-2 ${stat.color}`} />
                        {stat.title}
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="text-2xl font-bold text-foreground">{stat.value}</div>
                      <p className="text-sm text-muted-foreground">{stat.description}</p>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </section>

            {/* Executive Modules */}
            <section aria-labelledby="modules-heading">
              <h3 id="modules-heading" className="text-lg font-semibold mb-4 text-foreground">
                Executive Tools
              </h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                {executiveModules.map((module) => (
                  <Card 
                    key={module.id} 
                    className="cursor-pointer transition-all duration-300 hover:shadow-strong transform hover:scale-105 border-border/50"
                    onClick={() => handleModuleClick(module.id)}
                    onKeyDown={(e) => {
                      if (e.key === 'Enter' || e.key === ' ') {
                        e.preventDefault();
                        handleModuleClick(module.id);
                      }
                    }}
                    tabIndex={0}
                    role="button"
                    aria-label={`${module.title}: ${module.description}`}
                  >
                    <CardHeader className="pb-3">
                      <div className="flex items-start justify-between">
                        <div className="flex items-center space-x-3">
                          <div className={`w-12 h-12 ${module.gradient} rounded-lg flex items-center justify-center shadow-medium`}>
                            <module.icon className="w-6 h-6 text-white" />
                          </div>
                          <div>
                            <CardTitle className="text-lg text-foreground">{module.title}</CardTitle>
                            <Badge variant="default" className="text-xs mt-1 rounded-sm bg-violet-800 text-white">
                              Active
                            </Badge>
                          </div>
                        </div>
                      </div>
                    </CardHeader>
                    <CardContent>
                      <CardDescription className="text-muted-foreground mb-3">
                        {module.description}
                      </CardDescription>
                      <div className="flex items-center justify-between pt-2 border-t border-border/50">
                        <span className="text-sm text-muted-foreground">{module.stats.label}</span>
                        <div className="flex items-center space-x-1">
                          <span className="font-medium text-foreground">{module.stats.value}</span>
                          {module.stats.trend === 'up' && <TrendingUp className="w-4 h-4 text-secondary-accent" />}
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </section>
          </TabsContent>

          <TabsContent value="decisions" className="space-y-6">
            <section aria-labelledby="decisions-heading">
              <h2 id="decisions-heading" className="text-lg font-semibold mb-4 text-foreground">
                Decision Tracking & Follow-up
              </h2>
              <div className="space-y-4">
                {recentDecisions.map((decision) => (
                  <Card key={decision.id} className="border-border/50">
                    <CardContent className="p-6">
                      <div className="flex items-start justify-between">
                        <div className="flex-1">
                          <h4 className="font-semibold text-foreground mb-2">{decision.title}</h4>
                          <div className="flex items-center space-x-4 text-sm text-muted-foreground">
                            <span>Date: {decision.date}</span>
                            <span>Follow-up: {decision.followUp}</span>
                          </div>
                        </div>
                        <div className="flex items-center space-x-2">
                          {getPriorityBadge(decision.priority)}
                          <Badge 
                            variant="outline" 
                            className={getStatusColor(decision.status)}
                          >
                            {decision.status}
                          </Badge>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </section>
          </TabsContent>

          <TabsContent value="meetings" className="space-y-6">
            <section aria-labelledby="meetings-heading">
              <h2 id="meetings-heading" className="text-lg font-semibold mb-4 text-foreground">
                Upcoming Meetings & Approvals
              </h2>
              <div className="space-y-4">
                {upcomingMeetings.map((meeting) => (
                  <Card key={meeting.id} className="border-border/50">
                    <CardContent className="p-6">
                      <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-4">
                          <div className="w-12 h-12 bg-gradient-primary rounded-lg flex items-center justify-center">
                            <Calendar className="w-6 h-6 text-white" />
                          </div>
                          <div>
                            <h4 className="font-semibold text-foreground">{meeting.title}</h4>
                            <div className="flex items-center space-x-4 text-sm text-muted-foreground">
                              <span>{meeting.date} at {meeting.time}</span>
                              <span>{meeting.attendees} attendees</span>
                            </div>
                          </div>
                        </div>
                        <Badge variant="secondary" className="capitalize">
                          {meeting.type}
                        </Badge>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </section>
          </TabsContent>

          <TabsContent value="reports" className="space-y-6">
            <section aria-labelledby="reports-heading">
              <h2 id="reports-heading" className="text-lg font-semibold mb-4 text-foreground">
                Executive Reports & Analytics
              </h2>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <Card className="border-border/50">
                  <CardHeader>
                    <CardTitle className="flex items-center">
                      <BarChart3 className="w-5 h-5 mr-2" />
                      Performance Metrics
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-4">
                      <div>
                        <div className="flex justify-between text-sm mb-1">
                          <span>Strategic Goals</span>
                          <span>73%</span>
                        </div>
                        <Progress value={73} className="h-2" />
                      </div>
                      <div>
                        <div className="flex justify-between text-sm mb-1">
                          <span>Operational Efficiency</span>
                          <span>85%</span>
                        </div>
                        <Progress value={85} className="h-2" />
                      </div>
                      <div>
                        <div className="flex justify-between text-sm mb-1">
                          <span>Stakeholder Satisfaction</span>
                          <span>92%</span>
                        </div>
                        <Progress value={92} className="h-2" />
                      </div>
                    </div>
                  </CardContent>
                </Card>

                <Card className="border-border/50">
                  <CardHeader>
                    <CardTitle className="flex items-center">
                      <FileText className="w-5 h-5 mr-2" />
                      Recent Reports
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-3">
                      <div className="flex justify-between items-center">
                        <span className="text-sm">Q4 Performance Review</span>
                        <Badge variant="outline">Ready</Badge>
                      </div>
                      <div className="flex justify-between items-center">
                        <span className="text-sm">Board Meeting Minutes</span>
                        <Badge variant="outline">Draft</Badge>
                      </div>
                      <div className="flex justify-between items-center">
                        <span className="text-sm">Strategic Plan Update</span>
                        <Badge variant="outline">In Progress</Badge>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              </div>
            </section>
          </TabsContent>
        </Tabs>
      </main>
    </div>
  );
}