import React, { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import {
  Tabs,
  TabsList,
  TabsTrigger,
  TabsContent,
} from "@/components/ui/tabs";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { BookOpen, TrendingUp, Calendar, Home, LogOut } from "lucide-react";

import { DailyJournal } from "@/components/DailyJournal";
import { MonthlyReview } from "@/components/MonthlyReview";
import BudgetTracker from "@/components/BudgetTracker";

const Dashboard: React.FC = () => {
  const [activeTab, setActiveTab] = useState("dashboard");
  const [userEmail, setUserEmail] = useState<string | null>(null);
  const navigate = useNavigate();

  useEffect(() => {
    const stored = localStorage.getItem("user");
    if (stored) {
      const parsed = JSON.parse(stored);
      setUserEmail(parsed.email);
    } else {
      navigate("/login");
    }
  }, [navigate]);

  const handleLogout = () => {
    localStorage.removeItem("user");
    navigate("/login");
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 via-indigo-50 to-purple-50">
      <div className="container mx-auto px-4 py-6 max-w-4xl">
        {/* Header */}
        <div className="flex items-center justify-between mb-6">
          <div>
            <h1 className="text-3xl font-bold bg-gradient-to-r from-indigo-600 to-purple-600 bg-clip-text text-transparent">
              Personal Development Track - 2025
            </h1>
            <p className="text-gray-600 text-sm">
              Welcome, <span className="font-semibold">{userEmail}</span>
            </p>
          </div>
          <Button variant="outline" onClick={handleLogout} className="flex items-center gap-2">
            <LogOut size={16} />
            Logout
          </Button>
        </div>

        {/* Navigation Tabs */}
        <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
          <TabsList className="grid w-full grid-cols-4 mb-4 bg-white/70 backdrop-blur-sm rounded-lg shadow-sm">
            <TabsTrigger value="dashboard" className="flex items-center gap-2">
              <Home size={16} />
              <span className="hidden sm:inline">Dashboard</span>
            </TabsTrigger>
            <TabsTrigger value="journal" className="flex items-center gap-2">
              <BookOpen size={16} />
              <span className="hidden sm:inline">Journal</span>
            </TabsTrigger>
            <TabsTrigger value="review" className="flex items-center gap-2">
              <Calendar size={16} />
              <span className="hidden sm:inline">Review</span>
            </TabsTrigger>
            <TabsTrigger value="budget" className="flex items-center gap-2">
              <TrendingUp size={16} />
              <span className="hidden sm:inline">Budget</span>
            </TabsTrigger>
          </TabsList>

          <TabsContent value="dashboard">
            <Card className="bg-white/80 backdrop-blur-sm shadow-lg">
              <CardHeader>
                <CardTitle className="text-xl font-semibold">Your Progress Overview</CardTitle>
              </CardHeader>
              <CardContent>
                <p className="text-gray-700">
                  Here you can track your personal development progress, set goals, and monitor your achievements.
                </p>
                {/* Additional dashboard content can go here */}
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="journal">
            <DailyJournal />
          </TabsContent>

          <TabsContent value="review">
            <MonthlyReview />
          </TabsContent>

          <TabsContent value="budget">
            <BudgetTracker />
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
};

export default Dashboard;
