import { useState, useEffect } from "react";
import { supabase } from "@/components/supabaseClient";
import {
  Card,
  CardContent,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Heart, Star, Target, Lightbulb, Calendar, Quote } from "lucide-react";
import { format } from "date-fns";
import { useNavigate } from "react-router-dom";

export const DailyJournal = () => {
  const navigate = useNavigate();
  const [userId, setUserId] = useState<string | null>(null);
  const [selectedMood, setSelectedMood] = useState("");
  const [gratitudeItems, setGratitudeItems] = useState([""]);
  const [wins, setWins] = useState("");
  const [challenges, setChallenges] = useState("");
  const [learnings, setLearnings] = useState("");
  const [intentions, setIntentions] = useState("");
  const [reflection, setReflection] = useState("");
  const [customQuote, setCustomQuote] = useState("");
  const [existingEntry, setExistingEntry] = useState<any>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [isSaving, setIsSaving] = useState(false);

  const moods = [
    { emoji: "😊", label: "Happy", value: "happy" },
    { emoji: "😌", label: "Calm", value: "calm" },
    { emoji: "🤔", label: "Thoughtful", value: "thoughtful" },
    { emoji: "😤", label: "Determined", value: "determined" },
    { emoji: "😔", label: "Low", value: "low" },
    { emoji: "😴", label: "Tired", value: "tired" },
  ];

  useEffect(() => {
    const fetchUserAndEntry = async () => {
      setIsLoading(true);
      try {
        // Get current user
        const { data: { user }, error: userError } = await supabase.auth.getUser();
        if (userError) throw userError;
        setUserId(user?.id || null);

        if (user?.id) {
          // Check for existing entry today using UTC for consistent timezone handling
          const today = new Date();
          const todayUTC = new Date(Date.UTC(today.getFullYear(), today.getMonth(), today.getDate()));
          const tomorrowUTC = new Date(todayUTC);
          tomorrowUTC.setDate(tomorrowUTC.getDate() + 1);

          const { data: existingEntries, error: fetchError } = await supabase
            .from("daily_journals")
            .select("*")
            .eq("user_id", user.id)
            .gte("date", todayUTC.toISOString().split('T')[0]) // Use date column instead of created_at
            .lt("date", tomorrowUTC.toISOString().split('T')[0]);

          if (fetchError) throw fetchError;

          if (existingEntries && existingEntries.length > 0) {
            setExistingEntry(existingEntries[0]);
          }
        }
      } catch (error) {
        console.error("Error fetching data:", error);
      } finally {
        setIsLoading(false);
      }
    };

    fetchUserAndEntry();
  }, [navigate]);

  const addGratitudeItem = () => {
    setGratitudeItems([...gratitudeItems, ""]);
  };

  const updateGratitudeItem = (index: number, value: string) => {
    const updated = [...gratitudeItems];
    updated[index] = value;
    setGratitudeItems(updated);
  };

  const removeGratitudeItem = (index: number) => {
    if (gratitudeItems.length > 1) {
      setGratitudeItems(gratitudeItems.filter((_, i) => i !== index));
    }
  };

  const handleSave = async () => {
    if (!userId) {
      alert("You must be logged in to save your journal entry.");
      return;
    }

    if (existingEntry) {
      alert("You've already created a journal entry for today.");
      return;
    }

    // Form validation
    if (!selectedMood) {
      alert("Please select your mood for today.");
      return;
    }

    const validGratitudeItems = gratitudeItems.filter((item) => item.trim() !== "");
    if (validGratitudeItems.length === 0) {
      alert("Please add at least one gratitude item.");
      return;
    }

    setIsSaving(true);
    try {
      const today = new Date();
      const todayDateString = today.toISOString().split('T')[0]; // Format as YYYY-MM-DD
      
      const journalEntry = {
        user_id: userId,
        date: todayDateString, // Add the date field
        mood: selectedMood,
        gratitude: validGratitudeItems,
        wins: wins.trim(),
        challenges: challenges.trim(),
        learnings: learnings.trim(),
        intentions: intentions.trim(),
        reflection: reflection.trim(),
        quote: customQuote.trim(),
      };

      console.log("Saving entry:", journalEntry);

      const { data, error } = await supabase
        .from("daily_journals")
        .insert([journalEntry])
        .select();

      if (error) {
        console.error("Supabase error details:", error);
        throw error;
      }

      console.log("Saved successfully:", data);
      
      resetForm();
      alert("Journal entry saved successfully!");
      navigate("/IndividualDashboard");
    } catch (error: any) {
      console.error("Full error object:", error);
      alert(`Failed to save journal entry: ${error.message}`);
    } finally {
      setIsSaving(false);
    }
  };

  const resetForm = () => {
    setSelectedMood("");
    setGratitudeItems([""]);
    setWins("");
    setChallenges("");
    setLearnings("");
    setIntentions("");
    setReflection("");
    setCustomQuote("");
  };

  if (isLoading) {
    return (
      <div className="flex justify-center items-center h-screen">
        <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-indigo-500"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6 max-w-4xl mx-auto pb-10 px-4">
      <Button
        variant="outline"
        onClick={() => navigate("/IndividualDashboard")}
        className="fixed top-4 left-4 z-50 bg-white shadow-md hover:bg-muted text-sm"
      >
        ← Back to Dashboard
      </Button>

      {existingEntry ? (
        <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
          <CardHeader>
            <CardTitle className="flex items-center gap-2 text-indigo-600">
              <Heart className="h-5 w-5" />
              You've already journaled today!
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div>
              <h3 className="font-medium">Date:</h3>
              <p>{existingEntry.date}</p>
            </div>
            <div>
              <h3 className="font-medium">Mood:</h3>
              <p className="capitalize">{existingEntry.mood}</p>
            </div>
            <div>
              <h3 className="font-medium">Gratitude:</h3>
              <ul className="list-disc pl-5">
                {existingEntry.gratitude.map((item: string, index: number) => (
                  <li key={index}>{item}</li>
                ))}
              </ul>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <h3 className="font-medium">Wins:</h3>
                <p>{existingEntry.wins}</p>
              </div>
              <div>
                <h3 className="font-medium">Challenges:</h3>
                <p>{existingEntry.challenges}</p>
              </div>
              <div>
                <h3 className="font-medium">Learnings:</h3>
                <p>{existingEntry.learnings}</p>
              </div>
              <div>
                <h3 className="font-medium">Intentions:</h3>
                <p>{existingEntry.intentions}</p>
              </div>
            </div>
            {existingEntry.reflection && (
              <div>
                <h3 className="font-medium">Reflection:</h3>
                <p>{existingEntry.reflection}</p>
              </div>
            )}
            {existingEntry.quote && (
              <div>
                <h3 className="font-medium">Quote:</h3>
                <p className="italic">"{existingEntry.quote}"</p>
              </div>
            )}
            <Button 
              onClick={() => navigate("/IndividualDashboard")}
              className="w-full mt-4"
            >
              Back to Dashboard
            </Button>
          </CardContent>
        </Card>
      ) : (
        <>
          <Card className="bg-gradient-to-r from-purple-500 to-pink-500 text-white border-0">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Heart className="h-6 w-6" />
                Daily Journal - {format(new Date(), "MMMM do, yyyy")}
              </CardTitle>
            </CardHeader>
          </Card>

          {/* Mood Section */}
          <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
            <CardHeader>
              <CardTitle className="text-lg">How are you feeling today? *</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-3 gap-3">
                {moods.map((mood) => (
                  <Button
                    key={mood.value}
                    variant={selectedMood === mood.value ? "default" : "outline"}
                    className={`p-4 h-auto ${
                      selectedMood === mood.value
                        ? "bg-indigo-500 hover:bg-indigo-600"
                        : ""
                    }`}
                    onClick={() => setSelectedMood(mood.value)}
                  >
                    <div className="text-center">
                      <div className="text-2xl mb-1">{mood.emoji}</div>
                      <div className="text-sm">{mood.label}</div>
                    </div>
                  </Button>
                ))}
              </div>
            </CardContent>
          </Card>

          {/* Gratitude Section */}
          <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Star className="h-5 w-5" />
                What am I grateful for today? *
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              {gratitudeItems.map((item, index) => (
                <div key={index} className="flex gap-2">
                  <Input
                    placeholder={`Gratitude #${index + 1}`}
                    value={item}
                    onChange={(e) => updateGratitudeItem(index, e.target.value)}
                    className="flex-1"
                  />
                  {gratitudeItems.length > 1 && (
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => removeGratitudeItem(index)}
                    >
                      ×
                    </Button>
                  )}
                </div>
              ))}
              <Button variant="outline" onClick={addGratitudeItem} className="w-full">
                + Add Another
              </Button>
            </CardContent>
          </Card>

          {/* Wins & Challenges */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
              <CardHeader>
                <CardTitle className="flex items-center gap-2 text-green-600">
                  <Target className="h-5 w-5" />
                  Wins of the Day
                </CardTitle>
              </CardHeader>
              <CardContent>
                <Textarea
                  placeholder="What went well today? What are you proud of?"
                  value={wins}
                  onChange={(e) => setWins(e.target.value)}
                  className="min-h-24"
                />
              </CardContent>
            </Card>

            <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
              <CardHeader>
                <CardTitle className="flex items-center gap-2 text-orange-600">
                  <Calendar className="h-5 w-5" />
                  Challenges Faced
                </CardTitle>
              </CardHeader>
              <CardContent>
                <Textarea
                  placeholder="What challenges did you encounter? How did you handle them?"
                  value={challenges}
                  onChange={(e) => setChallenges(e.target.value)}
                  className="min-h-24"
                />
              </CardContent>
            </Card>
          </div>

          {/* Learnings & Intentions */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
              <CardHeader>
                <CardTitle className="flex items-center gap-2 text-blue-600">
                  <Lightbulb className="h-5 w-5" />
                  What I Learned Today
                </CardTitle>
              </CardHeader>
              <CardContent>
                <Textarea
                  placeholder="What insights or lessons did you gain today?"
                  value={learnings}
                  onChange={(e) => setLearnings(e.target.value)}
                  className="min-h-24"
                />
              </CardContent>
            </Card>

            <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
              <CardHeader>
                <CardTitle className="flex items-center gap-2 text-purple-600">
                  <Target className="h-5 w-5" />
                  Intentions for Tomorrow
                </CardTitle>
              </CardHeader>
              <CardContent>
                <Textarea
                  placeholder="What do you want to focus on tomorrow?"
                  value={intentions}
                  onChange={(e) => setIntentions(e.target.value)}
                  className="min-h-24"
                />
              </CardContent>
            </Card>
          </div>

          {/* Reflection */}
          <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Heart className="h-5 w-5" />
                Daily Reflection
              </CardTitle>
            </CardHeader>
            <CardContent>
              <Textarea
                placeholder="Take a moment to reflect on your day. How do you feel? What patterns do you notice?"
                value={reflection}
                onChange={(e) => setReflection(e.target.value)}
                className="min-h-32"
              />
            </CardContent>
          </Card>

          {/* Quote */}
          <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Quote className="h-5 w-5" />
                Quote or Mantra for the Day
              </CardTitle>
            </CardHeader>
            <CardContent>
              <Input
                placeholder="Enter your personal quote or mantra for today..."
                value={customQuote}
                onChange={(e) => setCustomQuote(e.target.value)}
              />
              <div className="mt-3 p-3 bg-indigo-50 rounded-lg">
                <p className="text-sm text-indigo-700 italic">
                  "Success is not final, failure is not fatal: it is the courage to continue that counts." - Winston Churchill
                </p>
                <p className="text-xs text-indigo-500 mt-1">Suggested quote of the day</p>
              </div>
            </CardContent>
          </Card>

          {/* Save */}
          <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
            <CardContent className="pt-6">
              <Button
                onClick={handleSave}
                disabled={isSaving}
                className="w-full bg-gradient-to-r from-indigo-500 to-purple-600 hover:from-indigo-600 hover:to-purple-700 text-white font-medium py-3"
              >
                {isSaving ? "Saving..." : "Save Today's Journal Entry"}
              </Button>
              <p className="text-xs text-gray-500 mt-2 text-center">* indicates required fields</p>
            </CardContent>
          </Card>
        </>
      )}
    </div>
  );
};