import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { supabase } from '@/components/supabaseClient';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardHeader, CardTitle, CardDescription, CardContent, CardFooter } from '@/components/ui/card';
import toast, { Toaster } from 'react-hot-toast';
import { Building2, Shield, BadgeCheck, Mail, RefreshCw } from 'lucide-react';

const CreateOrganization = () => {
  const [orgName, setOrgName] = useState('');
  const [orgCode, setOrgCode] = useState('');
  const [orgType, setOrgType] = useState<'tools' | 'leadership'>('tools');
  const [loading, setLoading] = useState(false);
  const [generatingCode, setGeneratingCode] = useState(false);
  const navigate = useNavigate();

  const generateRandomCode = () => {
    const chars = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789'; // Exclude easily confused characters
    let result = '';
    for (let i = 0; i < 6; i++) {
      result += chars.charAt(Math.floor(Math.random() * chars.length));
    }
    return result;
  };

  const handleGenerateCode = async () => {
    setGeneratingCode(true);
    try {
      const { data: { user }, error } = await supabase.auth.getUser();
      if (error || !user?.email) throw new Error('User not authenticated');

      const newCode = generateRandomCode();
      setOrgCode(newCode);

      // Send email with the code (using Supabase email or your email service)
      const { error: emailError } = await supabase
        .from('email_templates')
        .insert({
          to: user.email,
          subject: 'Your Organization Access Code',
          html: `
            <p>Your organization access code is: <strong>${newCode}</strong></p>
            <p>Enter this code when creating your organization.</p>
          `
        });

      if (emailError) throw emailError;

      toast.success('Access code generated and sent to your email!');
    } catch (error: any) {
      toast.error(error.message || 'Failed to generate code');
    } finally {
      setGeneratingCode(false);
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      // Validate inputs
      if (!orgCode) throw new Error('Please generate an access code');
      if (!orgName) throw new Error('Organization name is required');

      const { data: { user }, error: userError } = await supabase.auth.getUser();
      if (userError || !user) throw new Error('User not authenticated');

      // Create organization with the code
      const { data: org, error: orgError } = await supabase
        .from('organizations')
        .insert({
          name: orgName,
          access_code: orgCode,
          organization_type: orgType,
          created_by: user.id
        })
        .select()
        .single();

      if (orgError) throw orgError;

      // Add user as admin
      const { error: memberError } = await supabase
        .from('organization_members')
        .insert({
          organization_id: org.id,
          user_id: user.id,
          user_name: user.user_metadata.full_name || user.email,
          user_email: user.email,
          role: 'admin'
        });

      if (memberError) throw memberError;

      toast.success('Organization created successfully!');
      setTimeout(() => navigate('/OrganizationDashboard'), 1500);
    } catch (error: any) {
      toast.error(error.message || 'Failed to create organization');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen flex items-center justify-center bg-gradient-to-b from-background to-muted/20 py-12 px-4">
      <Toaster position="top-center" />
      
      <Card className="w-full max-w-md">
        <CardHeader className="text-center">
          <Building2 className="w-12 h-12 mx-auto text-primary mb-4" />
          <CardTitle className="text-2xl">Create New Organization</CardTitle>
          <CardDescription>
            Set up your organization to access {orgType === 'tools' ? 'Organizational Tools' : 'Elective Leadership Suite'}
          </CardDescription>
        </CardHeader>
        
        <form onSubmit={handleSubmit}>
          <CardContent className="space-y-4">
            <div className="grid grid-cols-2 gap-4 mb-4">
              <Button
                type="button"
                variant={orgType === 'tools' ? 'default' : 'outline'}
                onClick={() => setOrgType('tools')}
              >
                <Shield className="w-4 h-4 mr-2" />
                Organizational Tools
              </Button>
              <Button
                type="button"
                variant={orgType === 'leadership' ? 'default' : 'outline'}
                onClick={() => setOrgType('leadership')}
              >
                <BadgeCheck className="w-4 h-4 mr-2" />
                Leadership Suite
              </Button>
            </div>

            <div className="space-y-2">
              <Label htmlFor="orgName">Organization Name</Label>
              <Input
                id="orgName"
                placeholder="Type your Organization Name"
                value={orgName}
                onChange={(e) => setOrgName(e.target.value)}
                required
              />
            </div>

            <div className="space-y-2">
              <Label>Organization Access Code</Label>
              <div className="flex gap-2">
                <Input
                  placeholder=""
                  value={orgCode}
                  readOnly
                  className="flex-1"
                />
                <Button
                  type="button"
                  onClick={handleGenerateCode}
                  disabled={generatingCode}
                >
                  {generatingCode ? (
                    <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                  ) : (
                    <Mail className="w-4 h-4 mr-2" />
                  )}
                  {generatingCode ? 'Generating...' : 'Get Code'}
                </Button>
              </div>
              <p className="text-xs text-muted-foreground">
                A 6-digit code will be sent to your email
              </p>
            </div>

            {orgType === 'leadership' && (
              <div className="bg-blue-50 p-4 rounded-md border border-blue-200">
                <h4 className="font-medium text-blue-800 flex items-center mb-2">
                  <BadgeCheck className="w-4 h-4 mr-2" />
                  Leadership Suite Includes:
                </h4>
                <ul className="text-sm text-blue-700 space-y-1">
                  <li>• Advanced member management</li>
                  <li>• Election tools</li>
                  <li>• Leadership analytics</li>
                  <li>• Custom roles and permissions</li>
                </ul>
              </div>
            )}

            {orgType === 'tools' && (
              <div className="bg-purple-50 p-4 rounded-md border border-purple-200">
                <h4 className="font-medium text-purple-800 flex items-center mb-2">
                  <Shield className="w-4 h-4 mr-2" />
                  Organizational Tools Include:
                </h4>
                <ul className="text-sm text-purple-700 space-y-1">
                  <li>• Team dashboards</li>
                  <li>• KPI tracking</li>
                  <li>• Organizational health metrics</li>
                  <li>• Collaboration tools</li>
                </ul>
              </div>
            )}
          </CardContent>
          <CardFooter className="flex flex-col gap-2">
            <Button type="submit" className="w-full" disabled={loading || !orgCode}>
              {loading ? 'Creating...' : 'Create Organization'}
            </Button>
            <Button
              type="button"
              variant="outline"
              className="w-full"
              onClick={() => navigate("/IndividualDashboard")}
              disabled={loading}
            >
              Cancel
            </Button>
          </CardFooter>
        </form>
      </Card>
    </div>
  );
};

export default CreateOrganization;