import React, { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { ArrowLeft, Brain, Heart, AlertTriangle, Clock, Battery, Thermometer, Activity } from 'lucide-react';

interface BurnoutTrackingProps {
  onBack: () => void;
}

const burnoutData = [
  {
    id: 1,
    employee: 'John Smith',
    department: 'Development',
    burnoutScore: 85,
    riskLevel: 'Low',
    workHours: 42,
    stressLevel: 3,
    satisfaction: 8,
    lastAssessment: '2025-01-15'
  },
  {
    id: 2,
    employee: 'Sarah Johnson',
    department: 'Design',
    burnoutScore: 72,
    riskLevel: 'Medium',
    workHours: 45,
    stressLevel: 6,
    satisfaction: 7,
    lastAssessment: '2025-01-14'
  },
  {
    id: 3,
    employee: 'Mike Chen',
    department: 'Sales',
    burnoutScore: 45,
    riskLevel: 'High',
    workHours: 52,
    stressLevel: 8,
    satisfaction: 4,
    lastAssessment: '2025-01-13'
  },
  {
    id: 4,
    employee: 'Lisa Garcia',
    department: 'Marketing',
    burnoutScore: 78,
    riskLevel: 'Low',
    workHours: 38,
    stressLevel: 4,
    satisfaction: 8,
    lastAssessment: '2025-01-12'
  }
];

const wellnessMetrics = {
  averageBurnoutScore: 70,
  highRiskEmployees: 1,
  mediumRiskEmployees: 1,
  lowRiskEmployees: 2,
  averageWorkHours: 44,
  averageStressLevel: 5.2,
  averageSatisfaction: 6.8
};

const getRiskBadge = (risk: string) => {
  switch (risk) {
    case 'Low':
      return <Badge className="bg-accent-bright text-primary">Low Risk</Badge>;
    case 'Medium':
      return <Badge className="bg-secondary-accent text-primary">Medium Risk</Badge>;
    case 'High':
      return <Badge className="bg-destructive text-destructive-foreground">High Risk</Badge>;
    default:
      return <Badge variant="secondary">{risk}</Badge>;
  }
};

const getScoreColor = (score: number) => {
  if (score >= 75) return 'text-accent-bright';
  if (score >= 50) return 'text-secondary-accent';
  return 'text-destructive';
};

const departments = ['All', 'Development', 'Design', 'Sales', 'Marketing', 'HR'];

export function BurnoutTracking({ onBack }: BurnoutTrackingProps) {
  const [selectedDepartment, setSelectedDepartment] = useState<string>('All');

  const filteredData = selectedDepartment === 'All' 
    ? burnoutData 
    : burnoutData.filter(emp => emp.department === selectedDepartment);

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="border-b bg-card/50 backdrop-blur-sm sticky top-0 z-50">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-4">
              <Button variant="ghost" size="sm" onClick={onBack}>
                <ArrowLeft className="w-4 h-4 mr-2" />
                Back to Organizational Tools
              </Button>
              <div className="w-10 h-10 rounded-lg bg-gradient-wellness flex items-center justify-center">
                <Brain className="w-6 h-6 text-white" />
              </div>
              <div>
                <h1 className="text-2xl font-bold bg-gradient-wellness bg-clip-text text-transparent">
                  Burnout Tracking
                </h1>
                <p className="text-sm text-muted-foreground">Monitor employee wellness and prevent burnout</p>
              </div>
            </div>
            <Badge className="bg-gradient-primary text-white">Wellness Monitor</Badge>
          </div>
        </div>
      </div>

      {/* Main Content */}
      <div className="container mx-auto px-6 py-8">
        {/* Wellness Overview Cards */}
        <div className="mb-8 animate-fade-in">
          <h2 className="text-lg font-semibold mb-4 text-foreground">Wellness Overview</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            <Card className="border-border/50">
              <CardHeader className="pb-3">
                <CardTitle className="text-sm font-medium text-muted-foreground flex items-center">
                  <Battery className="w-4 h-4 mr-2" />
                  Avg Burnout Score
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="text-3xl font-bold text-foreground">{wellnessMetrics.averageBurnoutScore}</div>
                <p className="text-xs text-muted-foreground mt-1">Out of 100 (wellness scale)</p>
              </CardContent>
            </Card>

            <Card className="border-border/50">
              <CardHeader className="pb-3">
                <CardTitle className="text-sm font-medium text-muted-foreground flex items-center">
                  <AlertTriangle className="w-4 h-4 mr-2" />
                  High Risk
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="text-3xl font-bold text-destructive">{wellnessMetrics.highRiskEmployees}</div>
                <p className="text-xs text-muted-foreground mt-1">Employees needing attention</p>
              </CardContent>
            </Card>

            <Card className="border-border/50">
              <CardHeader className="pb-3">
                <CardTitle className="text-sm font-medium text-muted-foreground flex items-center">
                  <Clock className="w-4 h-4 mr-2" />
                  Avg Work Hours
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="text-3xl font-bold text-foreground">{wellnessMetrics.averageWorkHours}h</div>
                <p className="text-xs text-muted-foreground mt-1">Per week</p>
              </CardContent>
            </Card>

            <Card className="border-border/50">
              <CardHeader className="pb-3">
                <CardTitle className="text-sm font-medium text-muted-foreground flex items-center">
                  <Heart className="w-4 h-4 mr-2" />
                  Satisfaction
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="text-3xl font-bold text-foreground">{wellnessMetrics.averageSatisfaction}/10</div>
                <p className="text-xs text-muted-foreground mt-1">Average rating</p>
              </CardContent>
            </Card>
          </div>
        </div>

        {/* Risk Distribution */}
        <div className="mb-8 animate-slide-up" style={{ animationDelay: '0.2s' }}>
          <h3 className="text-lg font-semibold mb-4 text-foreground">Risk Distribution</h3>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            <Card className="border-border/50">
              <CardHeader className="pb-3">
                <CardTitle className="text-base font-semibold text-foreground flex items-center">
                  <div className="w-3 h-3 bg-accent-bright rounded-full mr-2"></div>
                  Low Risk
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold text-accent-bright">{wellnessMetrics.lowRiskEmployees}</div>
                <p className="text-sm text-muted-foreground">Healthy work-life balance</p>
              </CardContent>
            </Card>

            <Card className="border-border/50">
              <CardHeader className="pb-3">
                <CardTitle className="text-base font-semibold text-foreground flex items-center">
                  <div className="w-3 h-3 bg-secondary-accent rounded-full mr-2"></div>
                  Medium Risk
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold text-secondary-accent">{wellnessMetrics.mediumRiskEmployees}</div>
                <p className="text-sm text-muted-foreground">Monitor closely</p>
              </CardContent>
            </Card>

            <Card className="border-border/50">
              <CardHeader className="pb-3">
                <CardTitle className="text-base font-semibold text-foreground flex items-center">
                  <div className="w-3 h-3 bg-destructive rounded-full mr-2"></div>
                  High Risk
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold text-destructive">{wellnessMetrics.highRiskEmployees}</div>
                <p className="text-sm text-muted-foreground">Immediate intervention needed</p>
              </CardContent>
            </Card>
          </div>
        </div>

        {/* Employee Burnout Details */}
        <div className="animate-slide-up" style={{ animationDelay: '0.4s' }}>
          <div className="flex items-center justify-between mb-4">
            <h3 className="text-lg font-semibold text-foreground">Employee Wellness Details</h3>
            <Select value={selectedDepartment} onValueChange={setSelectedDepartment}>
              <SelectTrigger className="w-48">
                <SelectValue placeholder="Filter by department" />
              </SelectTrigger>
              <SelectContent>
                {departments.map((dept) => (
                  <SelectItem key={dept} value={dept}>{dept}</SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <Card className="border-border/50">
            <CardContent className="p-6">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Employee</TableHead>
                    <TableHead>Department</TableHead>
                    <TableHead>Burnout Score</TableHead>
                    <TableHead>Risk Level</TableHead>
                    <TableHead>Work Hours/Week</TableHead>
                    <TableHead>Stress Level</TableHead>
                    <TableHead>Satisfaction</TableHead>
                    <TableHead>Last Assessment</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredData.map((employee) => (
                    <TableRow key={employee.id}>
                      <TableCell className="font-medium">
                        <div className="flex items-center space-x-2">
                          <div className="w-8 h-8 rounded-full bg-gradient-primary flex items-center justify-center">
                            <span className="text-white text-xs font-medium">
                              {employee.employee.split(' ').map(n => n[0]).join('')}
                            </span>
                          </div>
                          <span>{employee.employee}</span>
                        </div>
                      </TableCell>
                      <TableCell>
                        <Badge variant="outline">{employee.department}</Badge>
                      </TableCell>
                      <TableCell>
                        <span className={`font-semibold ${getScoreColor(employee.burnoutScore)}`}>
                          {employee.burnoutScore}/100
                        </span>
                      </TableCell>
                      <TableCell>{getRiskBadge(employee.riskLevel)}</TableCell>
                      <TableCell>{employee.workHours}h</TableCell>
                      <TableCell>
                        <div className="flex items-center space-x-1">
                          <Thermometer className="w-4 h-4 text-muted-foreground" />
                          <span>{employee.stressLevel}/10</span>
                        </div>
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center space-x-1">
                          <Heart className="w-4 h-4 text-muted-foreground" />
                          <span>{employee.satisfaction}/10</span>
                        </div>
                      </TableCell>
                      <TableCell className="text-muted-foreground">{employee.lastAssessment}</TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </CardContent>
          </Card>
        </div>

        {/* Wellness Recommendations */}
        <div className="mt-8 animate-slide-up" style={{ animationDelay: '0.6s' }}>
          <h3 className="text-lg font-semibold mb-4 text-foreground">Wellness Recommendations</h3>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <Card className="border-border/50">
              <CardHeader className="pb-3">
                <CardTitle className="text-base font-semibold text-foreground flex items-center">
                  <Activity className="w-5 h-5 mr-2 text-accent-bright" />
                  Immediate Actions
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-2">
                <p className="text-sm text-foreground">• Schedule 1-on-1 with Mike Chen (high risk)</p>
                <p className="text-sm text-foreground">• Review workload distribution for Sales team</p>
                <p className="text-sm text-foreground">• Implement stress management workshops</p>
                <p className="text-sm text-foreground">• Consider flexible work arrangements</p>
              </CardContent>
            </Card>

            <Card className="border-border/50">
              <CardHeader className="pb-3">
                <CardTitle className="text-base font-semibold text-foreground flex items-center">
                  <Heart className="w-5 h-5 mr-2 text-primary" />
                  Preventive Measures
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-2">
                <p className="text-sm text-foreground">• Regular wellness check-ins</p>
                <p className="text-sm text-foreground">• Mental health support programs</p>
                <p className="text-sm text-foreground">• Work-life balance initiatives</p>
                <p className="text-sm text-foreground">• Team building activities</p>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </div>
  );
}