import React, { useEffect, useState } from "react";
import { useRouter } from "next/router";
import { supabase } from "@/components/supabaseClient";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";

const AcceptInvite = () => {
  const router = useRouter();
  const [status, setStatus] = useState<"loading" | "success" | "invalid" | "error" | "need_signin">("loading");
  const [inviteData, setInviteData] = useState(null);
  const [userEmail, setUserEmail] = useState("");

  useEffect(() => {
    const checkInvite = async () => {
      const { token } = router.query;
      
      if (!token) return;

      try {
        // Verify invite exists and is valid
        const { data: invite, error } = await supabase
          .from("organization_invites")
          .select(`
            *,
            organizations(name)
          `)
          .eq("id", token)
          .eq("status", "pending")
          .gt("expires_at", new Date().toISOString())
          .single();

        if (error || !invite) {
          setStatus("invalid");
          return;
        }

        setInviteData(invite);
        setUserEmail(invite.email);

        // Check if user is signed in and email matches
        const { data: { user } } = await supabase.auth.getUser();
        
        if (!user) {
          setStatus("need_signin");
          return;
        }

        if (user.email !== invite.email) {
          // User is signed in but with different email
          await supabase.auth.signOut();
          setStatus("need_signin");
          return;
        }

        // If we reach here, user is properly authenticated
        await processInviteAcceptance(token as string, user.id, invite);

      } catch (error) {
        console.error("Error checking invite:", error);
        setStatus("error");
      }
    };

    checkInvite();
  }, [router]);

  const processInviteAcceptance = async (token: string, userId: string, invite: any) => {
    try {
      // Add user to organization
      const { error: memberError } = await supabase
        .from("organization_members")
        .insert({
          user_id: userId,
          organization_id: invite.organization_id,
          role: invite.role,
        });

      if (memberError) {
        console.error("Member error:", memberError);
        setStatus("error");
        return;
      }

      // Update invite status
      const { error: updateError } = await supabase
        .from("organization_invites")
        .update({ 
          status: "accepted", 
          accepted_at: new Date().toISOString(),
          auth_user_id: userId
        })
        .eq("id", token);

      if (updateError) {
        console.error("Update error:", updateError);
      }

      setStatus("success");
      setTimeout(() => router.push("/dashboard"), 2000);

    } catch (error) {
      console.error("Error processing invite:", error);
      setStatus("error");
    }
  };

  const handleSignIn = async () => {
    // Redirect to sign in with the invite email pre-filled
    await router.push(`/auth/signin?email=${encodeURIComponent(userEmail)}&invite_token=${router.query.token}`);
  };

  const handleSignUp = async () => {
    // Redirect to sign up with the invite email pre-filled
    await router.push(`/auth/signup?email=${encodeURIComponent(userEmail)}&invite_token=${router.query.token}`);
  };

  if (status === "loading") {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <Card className="w-96">
          <CardContent className="pt-6">
            <div className="text-center">
              <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500 mx-auto mb-4"></div>
              <p>Processing your invitation...</p>
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  if (status === "need_signin" && inviteData) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <Card className="w-96">
          <CardHeader>
            <CardTitle>Join {inviteData.organizations.name}</CardTitle>
          </CardHeader>
          <CardContent>
            <p className="mb-4">You've been invited to join as a {inviteData.role}.</p>
            <p className="mb-6 text-sm text-muted-foreground">
              Please sign in or create an account with <strong>{userEmail}</strong> to accept this invitation.
            </p>
            <div className="space-y-3">
              <Button onClick={handleSignIn} className="w-full">
                Sign In
              </Button>
              <Button onClick={handleSignUp} variant="outline" className="w-full">
                Create Account
              </Button>
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  if (status === "success") {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <Card className="w-96">
          <CardContent className="pt-6">
            <div className="text-center">
              <div className="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <svg className="w-6 h-6 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M5 13l4 4L19 7" />
                </svg>
              </div>
              <h2 className="text-lg font-semibold mb-2">Invitation Accepted!</h2>
              <p className="text-muted-foreground mb-4">You've successfully joined the organization.</p>
              <Button onClick={() => router.push("/dashboard")}>
                Go to Dashboard
              </Button>
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="min-h-screen flex items-center justify-center">
      <Card className="w-96">
        <CardContent className="pt-6">
          <div className="text-center">
            <div className="w-12 h-12 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <svg className="w-6 h-6 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M6 18L18 6M6 6l12 12" />
              </svg>
            </div>
            <h2 className="text-lg font-semibold mb-2">
              {status === "invalid" ? "Invalid Invitation" : "Error"}
            </h2>
            <p className="text-muted-foreground mb-4">
              {status === "invalid" 
                ? "This invitation link is invalid or has expired." 
                : "An error occurred while processing your invitation."}
            </p>
            <Button onClick={() => router.push("/")}>
              Return Home
            </Button>
          </div>
        </CardContent>
      </Card>
    </div>
  );
};

export default AcceptInvite;